"""riboSeqR Galaxy unit tests"""
import sys
import unittest
import rpy2.robjects as robjects
from riboseqr import prepare, triplet, metagene, ribosome_profile, utils


R = robjects.r
DATA_DIR = 'test-data'
RIBO_FILES = ', '.join(
    '{0}/{1}'.format(DATA_DIR, item) for item in
    ['chlamy236_plus_deNovo_plusOnly_Index17',
     'chlamy236_plus_deNovo_plusOnly_Index3',
     'chlamy236_plus_deNovo_plusOnly_Index5',
     'chlamy236_plus_deNovo_plusOnly_Index7'])

RNA_FILES = ', '.join(
    '{0}/{1}'.format(DATA_DIR, item) for item in
    ['chlamy236_plus_deNovo_plusOnly_Index10',
     'chlamy236_plus_deNovo_plusOnly_Index12',
     'chlamy236_plus_deNovo_plusOnly_Index14',
     'chlamy236_plus_deNovo_plusOnly_Index16'])

FASTA_FILE = '{0}/{1}'.format(DATA_DIR, 'rsem_chlamy236_deNovo.transcripts.fa')


class PrepareTestCase(unittest.TestCase):

    def test_process_riboseqr_input(self):
        """Given riboSeqR format input files (bowtie output), save ribodata to \
        an R data file.

        """
        do_prepare()
        R('load("{}/Robjects.rda")'.format(DATA_DIR))
        ribodat, ribodat_ref = R['riboDat'], R['riboDat_REF']

        self.assertEqual(
            '{}'.format(ribodat), '{}'.format(ribodat_ref),
            'Generated RiboDat object must be equal to the reference')

    def test_process_args(self):
        """Test processing arguments. """
        # "ATG" -> c("ATG")
        rs = utils.process_args('ATG', ret_mode='charvector')
        self.assertEqual(rs, 'c("ATG")','Return string as a character vector.')

        # stop codons "TAG,TAA,TGA" -> c("TAG", "TAA", "TGA"). Also
        # replicate names, seqnames.
        rs = utils.process_args('TAG,TAA,TGA', ret_mode='charvector')
        self.assertEqual(
            rs, "c('TAG', 'TAA', 'TGA')",
            'Return comma separated strings as a character vector.')

        # "" -> None
        rs = utils.process_args('')
        self.assertIsNone(rs, 'Return empty string as None.')

        # "27,28" -> c(27, 28)
        rs = utils.process_args("27,28", ret_type='int', ret_mode='charvector')
        self.assertEqual(
            rs, "c(27, 28)", 'Return number strings as a character vector.')

        # "27,28" -> [27, 28]
        rs = utils.process_args("27,28", ret_type='int', ret_mode='list')
        self.assertEqual(rs, [27, 28], 'Return number strings as a list.')

        # "0,2" -> list(0,2)
        rs = utils.process_args("0,2", ret_type='int', ret_mode='listvector')
        self.assertEqual(
            rs, "list(0, 2)", 'Return number strings as a list vector.')

        # "50" -> 50
        rs = utils.process_args("50", ret_type='int')
        self.assertEqual(rs, 50, 'Return number string as a number.')

        # "-200" -> -200
        rs = utils.process_args("-200", ret_type='int')
        self.assertEqual(rs, -200, 'Return number string as a number.')

        # "TRUE" -> TRUE
        rs = utils.process_args("TRUE", ret_type='bool')
        self.assertEqual(rs, 'TRUE', 'Return bool string as bool.')

        # 'chlamy17,chlamy3' -> 'chlamy17,chlamy3' for ribo and rna names
        rs = utils.process_args('chlamy17,chlamy3')
        self.assertEqual(rs, 'chlamy17,chlamy3', 'Return csv string as string.')

        # 'chlamy17.idx, chlamy3.idx' -> ['chlamy17.idx', 'chlamy3.idx']
        rs = utils.process_args('chlamy17.idx, chlamy3.idx', ret_mode='list')
        self.assertEqual(rs, ['chlamy17.idx', 'chlamy3.idx'],
                         'Return files as a list.')


class TripletTestCase(unittest.TestCase):

    def test_find_periodicity(self):
        """Test triplet periodicity. """
        do_prepare()
        do_periodicity()
        fcs, fs, fasta_cds = R['fCs'], R['fS'], R['fastaCDS']

        R('load("{}/Robjects.rda")'.format(DATA_DIR))
        fcs_ref, fs_ref, fasta_cds_ref = (
            R['fCs_REF'], R['fS_REF'], R['fastaCDS_REF'])

        self.assertEqual(
            (str(fcs), str(fs), str(fasta_cds)),
            (str(fcs_ref), str(fs_ref), str(fasta_cds_ref)),
            msg='Generated fCs, fS and fastaCDS objects must be equal '
                'to reference')


class MetageneTestCase(unittest.TestCase):

    def test_do_analysis(self):
        """Test metagene analysis. """
        R('load("{}/Robjects.rda")'.format(DATA_DIR))
        do_prepare()
        do_periodicity()
        do_metagene()

        self.assertEqual(str(R['ffCs']),
                         str(R['ffCs_REF']), 'ffCs must be equal to reference')


def do_prepare():
    """Run the prepare step to generate riboDat from input files."""
    prepare.generate_ribodata(
        ribo_files=RIBO_FILES, rna_files=RNA_FILES, replicate_names='WT,WT,M,M',
        rdata_save='/tmp/Prepare.rda', sam_format=False, output_path="/tmp",
        html_file='/tmp/Prepare-report.html')


def do_periodicity():
    """Run the periodicity step"""
    triplet.find_periodicity(
        rdata_load='/tmp/Prepare.rda', start_codons='ATG',
        stop_codons='TAG,TAA,TGA', fasta_file=FASTA_FILE,
        include_lengths='25:30', analyze_plot_lengths='26:30',
        rdata_save='/tmp/Periodicity.rda', output_path="/tmp",
        html_file='/tmp/Periodicity-report.html')


def do_metagene():
    """Run the metagene step"""
    metagene.do_analysis(
        rdata_load='/tmp/Periodicity.rda', selected_lengths='27,28',
        selected_frames='1,0', hit_mean='50', unique_hit_mean='10',
        ratio_check='TRUE', min5p='-20', max5p='200', min3p='-200', max3p='20',
        cap='200', plot_title='Metagene analysis', plot_lengths='27',
        rdata_save='/tmp/Metagene.rda', output_path="/tmp",
        html_file='/tmp/Metagene-report.html')
