#!/usr/bin/env Rscript

args <- commandArgs(trailingOnly = TRUE)

library('latticeExtra')
library('colorRamps')

data.file <- read.table("SC_data.txt", sep="\t", header=TRUE, row.names=1) ### import spectral count data
data.file2 <- read.table("FDR_data.txt", sep="\t", header=TRUE, row.names=1) ### import FDR count data
data.file3 <- read.table("clustered_matrix.txt", sep="\t", header=TRUE, row.names=1) ### import clustered matrix
data.file4 <- scan("singletons.txt", what="", sep="\n", strip.white=T) ### import singleton data

#setting parameters

Sfirst=as.numeric(args[1]) #first FDR cutoff
Ssecond=as.numeric(args[2]) #second FDR cutoff
maxp=as.integer(args[3]) #maximum value for a spectral count

#calculate column and row lengths

#determine bait and prey ordering

bait_levels=names(data.file3)
prey_levels=c(rownames(data.file3),data.file4)

x_ord=factor(row.names(data.file),levels=prey_levels)
y_ord=factor(names(data.file),levels=bait_levels)

df<-data.frame(y=rep(y_ord,nrow(data.file))
	,x=rep(x_ord, each=ncol(data.file))
	,z1=as.vector(t(data.file)) # Circle color
	,z2=as.vector(t(data.file/apply(data.file,1,max))) # Circle size
	,z3=as.vector(t(data.file2)) # FDR
)
	
df$z1[df$z1>maxp] <- maxp #maximum value for spectral count
df$z2[df$z2==0] <- NA
df$z3[df$z3>Ssecond] <- 0.05*maxp
df$z3[df$z3<=Ssecond & df$z3>Sfirst] <- 0.5*maxp
df$z3[df$z3<=Sfirst] <- 1*maxp
df$z4 <- df$z1
df$z4[df$z4==0] <- 0
df$z4[df$z4>0] <- 2.5 

# The labeling for the colorkey

labelat = c(0, maxp)
labeltext = c(0, maxp)

# color scheme to use

nmb.colors<-maxp
z.colors<-grey(rev(seq(0,0.9,0.9/nmb.colors))) #grayscale color scale

#plot

pl <- levelplot(z1~x*y, data=df
	,col.regions =z.colors #terrain.colors(100)
	,scales = list(x = list(rot = 90), y=list(cex=0.8), tck=0) # rotates X,Y labels and changes scale 
	,colorkey = FALSE
	,xlab="Prey", ylab="Bait"
	,panel=function(x,y,z,...,col.regions){
		print(x)
		z.c<-df$z1[ (df$x %in% as.character(x)) & (df$y %in% y)]
		z.2<-df$z2[ (df$x %in% as.character(x)) & (df$y %in% y)]
		z.3<-df$z3
		z.4<-df$z4
		panel.xyplot(x,y
			,as.table=TRUE
			,pch=21 # point type to use (circles in this case)
			,cex=((z.2-min(z.2,na.rm=TRUE))/(max(z.2,na.rm=TRUE)-min(z.2,na.rm=TRUE)))*3 #circle size
			,fill=z.colors[floor((z.c-min(z.c,na.rm=TRUE))*nmb.colors/(max(z.c,na.rm=TRUE)-min(z.c,na.rm=TRUE)))+1] # circle colors
			,col=z.colors[1+z.3] # border colors
			,lex=z.4 #border thickness
			)
	}
	#,main="Fold change" # graph main title
	)
if(ncol(data.file) > 4) ht=3.5+(0.36*((ncol(data.file)-1)-4)) else ht=3.5
if(nrow(data.file) > 20) wd=8.25+(0.29*(nrow(data.file)-20)) else wd=5+(0.28*(nrow(data.file)-10))
pdf("dotplot.pdf", onefile = FALSE, paper = "special", height = ht, width = wd, pointsize = 2)
print(pl)
dev.off()
