
#  Copyright (c) 2003-2005 The Regents of the University of California.
#  Copyright (c) 2005 Gavin E. Crooks

#  This software is distributed under the MIT Open Source License.
#  <http://www.opensource.org/licenses/mit-license.html>
#
#  Permission is hereby granted, free of charge, to any person obtaining a 
#  copy of this software and associated documentation files (the "Software"),
#  to deal in the Software without restriction, including without limitation
#  the rights to use, copy, modify, merge, publish, distribute, sublicense,
#  and/or sell copies of the Software, and to permit persons to whom the
#  Software is furnished to do so, subject to the following conditions:
#
#  The above copyright notice and this permission notice shall be included
#  in all copies or substantial portions of the Software.
#
#  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
#  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
#  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
#  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
#  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
#  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN 
#  THE SOFTWARE.

""" Popular color codings for nucleic and amino acids. 

Classes:
    ColorScheme -- A color scheme
    ColorGroup  
    
    
Generic
    monochrome

Nucleotides
    nucleotide
    base pairing

Amino Acid
    hydrophobicity
    chemistry
    charge
    taylor

Status : Beta - Needs documentation.

"""
# Good online references include bioruby and the JalView alignment editor.
# Clamp, M., Cuff, J., Searle, S. M. and Barton, G. J. (2004), 
# "The Jalview Java Alignment Editor," Bioinformatics, 12, 426-7
# http://www.jalview.org

import sys

from corebio import seq
from color import Color
codon_alphabetU=['AAA', 'AAU', 'AAC', 'AAG', 'AUA', 'AUU', 'AUC', 'AUG', 'ACA', 'ACU', 'ACC', 'ACG', 'AGA', 'AGU', 'AGC', 'AGG', 'UAA', 'UAU', 'UAC', 'UAG', 'UUA', 'UUU', 'UUC', 'UUG', 'UCA', 'UCU', 'UCC', 'UCG', 'UGA', 'UGU', 'UGC', 'UGG', 'CAA', 'CAU', 'CAC', 'CAG', 'CUA', 'CUU', 'CUC', 'CUG', 'CCA', 'CCU', 'CCC', 'CCG', 'CGA', 'CGU', 'CGC', 'CGG', 'GAA', 'GAU', 'GAC', 'GAG', 'GUA', 'GUU', 'GUC', 'GUG', 'GCA', 'GCU', 'GCC', 'GCG', 'GGA', 'GGU', 'GGC', 'GGG']
codon_alphabetT=['AAA', 'AAT', 'AAC', 'AAG', 'ATA', 'ATT', 'ATC', 'ATG', 'ACA', 'ACT', 'ACC', 'ACG', 'AGA', 'AGT', 'AGC', 'AGG', 'TAA', 'TAT', 'TAC', 'TAG', 'TTA', 'TTT', 'TTC', 'TTG', 'TCA', 'TCT', 'TCC', 'TCG', 'TGA', 'TGT', 'TGC', 'TGG', 'CAA', 'CAT', 'CAC', 'CAG', 'CTA', 'CTT', 'CTC', 'CTG', 'CCA', 'CCT', 'CCC', 'CCG', 'CGA', 'CGT', 'CGC', 'CGG', 'GAA', 'GAT', 'GAC', 'GAG', 'GTA', 'GTT', 'GTC', 'GTG', 'GCA', 'GCT', 'GCC', 'GCG', 'GGA', 'GGT', 'GGC', 'GGG']

class ColorScheme(object):
    """ A coloring of an alphabet.
    
    title : string            -- A human readable description
    defualt_color : Color           --
    groups : list of color groups 
    alphabet : string               -- The set of colored symbols
    color -- A map between a symbol and a Coloring
    

    """
    
    def __init__(self, 
                groups = [], 
                title = "", 
                description = "",
                default_color = "black", 
                alphabet = seq.generic_alphabet) :
        """  """
        self.title= title
        self.description = description
        self.default_color = Color.from_string(default_color)
        self.groups = groups
        self.alphabet = alphabet
        #print >> sys.stderr, groups
	altype="codons"
	#print >> sys.stderr,altype
	#if(alphabet==codon_alphabet):
	  #print >> sys.stderr,"haleyulia it works"

        color = {}
        #print >> sys.stderr, groups
        if(alphabet!=codon_alphabetT and alphabet!=codon_alphabetU):
	  for cg in groups :
	      #print >> sys.stderr, cg
	      for s in cg.symbols :
		  color[s] = cg.color
		  #print >> sys.stderr, s
		  #print >> sys.stderr, cg
		  if s not in alphabet :
		      raise KeyError("Colored symbol does not exist in alphabet.")
	else:
	  for cg in groups :
	      #print >> sys.stderr, cg
	    color[cg.symbols] = cg.color
	    #print >> sys.stderr, cg.symbols
        self._color = color

    def color(self, symbol) :
        if symbol in self._color :
            return self._color[symbol]
        return self.default_color
        
class ColorGroup(object) :
    """Associate a group of symbols with a color"""
    def __init__(self, symbols, color, description=None) :
        self.symbols = symbols              
        self.color =  Color.from_string(color)
        self.description = description


         
monochrome = ColorScheme([]) # This list intentionally left blank
               
# From makelogo
nucleotide = ColorScheme([
    ColorGroup("G", "orange"),
    ColorGroup("TU", "red"),
    ColorGroup("C",  "blue"),
    ColorGroup("A",  "green") 
    ]) 

base_pairing = ColorScheme([
    ColorGroup("TAU",  "darkorange", "Weak (2 Watson-Crick hydrogen bonds)"),
    ColorGroup("GC",    "blue", "Strong (3 Watson-Crick hydrogen bonds)")],
    )


hydrophobicity = ColorScheme([
    ColorGroup( "RKDENQ",   "black", "hydrophobic"),
    ColorGroup( "SGHTAP",   "green", "neutral"  ),
    ColorGroup( "YVMCLFIW", "blue",  "hydrophilic") ],
    alphabet = seq.unambiguous_protein_alphabet
    )

# from makelogo
chemistry = ColorScheme([
  ColorGroup( "GSTYC",  "green",   "polar"),
  ColorGroup( "NQ",      "purple", "neutral"), 
  ColorGroup( "KRH",     "blue",   "basic"),
  ColorGroup( "DE",      "red",    "acidic"),
  ColorGroup("PAWFLIMV", "black",  "hydrophobic") ],
  alphabet = seq.unambiguous_protein_alphabet
  ) 
  

codonsU = ColorScheme([
ColorGroup( 'CAT', '#00FFFF'),
ColorGroup( 'CAU', '#00FFFF'),
ColorGroup( 'CAC', '#00FFFF'),

ColorGroup( 'AAA', '#00FFFF'),
ColorGroup( 'AAG', '#00FFFF'),

ColorGroup( 'CGT', '#00FFFF'),
ColorGroup( 'CGU', '#00FFFF'),
ColorGroup( 'CGC', '#00FFFF'),
ColorGroup( 'CGA', '#00FFFF'),
ColorGroup( 'CGG', '#00FFFF'),
ColorGroup( 'AGA', '#00FFFF'),
ColorGroup( 'AGG', '#00FFFF'),


ColorGroup( 'GAT', '#FF0000'),
ColorGroup( 'GAU', '#FF0000'),
ColorGroup( 'GAC', '#FF0000'),

ColorGroup( 'GAA', '#FF0000'),
ColorGroup( 'GAG', '#FF0000'),


ColorGroup( 'TCT', '#00FF00'),
ColorGroup( 'UCU', '#00FF00'),
ColorGroup( 'TCC', '#00FF00'),
ColorGroup( 'UCC', '#00FF00'),
ColorGroup( 'TCA', '#00FF00'),
ColorGroup( 'UCA', '#00FF00'),
ColorGroup( 'TCG', '#00FF00'),
ColorGroup( 'UCG', '#00FF00'),
ColorGroup( 'AGT', '#00FF00'),
ColorGroup( 'AGU', '#00FF00'),
ColorGroup( 'AGC', '#00FF00'),

ColorGroup( 'ACT', '#00FF00'),
ColorGroup( 'ACU', '#00FF00'),
ColorGroup( 'ACC', '#00FF00'),
ColorGroup( 'ACA', '#00FF00'),
ColorGroup( 'ACG', '#00FF00'),

ColorGroup( 'CAA', '#00FF00'),
ColorGroup( 'CAG', '#00FF00'),

ColorGroup( 'AAT', '#00FF00'),
ColorGroup( 'AAU', '#00FF00'),
ColorGroup( 'AAC', '#00FF00'),


ColorGroup( 'GCT', '#5555FF'),
ColorGroup( 'GCU', '#5555FF'),
ColorGroup( 'GCC', '#5555FF'),
ColorGroup( 'GCA', '#5555FF'),
ColorGroup( 'GCG', '#5555FF'),

ColorGroup( 'GTT', '#5555FF'),
ColorGroup( 'GUU', '#5555FF'),
ColorGroup( 'GTC', '#5555FF'),
ColorGroup( 'GUC', '#5555FF'),
ColorGroup( 'GTA', '#5555FF'),
ColorGroup( 'GUA', '#5555FF'),
ColorGroup( 'GTG', '#5555FF'),
ColorGroup( 'GUG', '#5555FF'),

ColorGroup( 'CTT', '#5555FF'),
ColorGroup( 'CUU', '#5555FF'),
ColorGroup( 'CTC', '#5555FF'),
ColorGroup( 'CUC', '#5555FF'),
ColorGroup( 'CTA', '#5555FF'),
ColorGroup( 'CUA', '#5555FF'),
ColorGroup( 'CTG', '#5555FF'),
ColorGroup( 'CUG', '#5555FF'),
ColorGroup( 'TTA', '#5555FF'),
ColorGroup( 'UUA', '#5555FF'),
ColorGroup( 'TTG', '#5555FF'),
ColorGroup( 'UUG', '#5555FF'),

ColorGroup( 'ATT', '#5555FF'),
ColorGroup( 'AUU', '#5555FF'),
ColorGroup( 'ATC', '#5555FF'),
ColorGroup( 'AUC', '#5555FF'),
ColorGroup( 'ATA', '#5555FF'),
ColorGroup( 'AUA', '#5555FF'),

ColorGroup( 'ATG', '#5555FF'),
ColorGroup( 'AUG', '#5555FF'),


ColorGroup( 'TTT', '#FF00FF'),
ColorGroup( 'UUU', '#FF00FF'),
ColorGroup( 'TTC', '#FF00FF'),
ColorGroup( 'UUC', '#FF00FF'),

ColorGroup( 'TAT', '#FF00FF'),
ColorGroup( 'UAU', '#FF00FF'),
ColorGroup( 'TAC', '#FF00FF'),
ColorGroup( 'UAC', '#FF00FF'),

ColorGroup( 'TGG', '#FF00FF'),
ColorGroup( 'UGG', '#FF00FF'),


ColorGroup( 'GGT', '#996600'),
ColorGroup( 'GGU', '#996600'),
ColorGroup( 'GGC', '#996600'),
ColorGroup( 'GGA', '#996600'),
ColorGroup( 'GGG', '#996600'),

ColorGroup( 'CCT', '#996600'),
ColorGroup( 'CCU', '#996600'),
ColorGroup( 'CCC', '#996600'),
ColorGroup( 'CCA', '#996600'),
ColorGroup( 'CCG', '#996600'),


ColorGroup( 'TGT', '#FFFF00'),
ColorGroup( 'UGU', '#FFFF00'),
ColorGroup( 'TGC', '#FFFF00'),
ColorGroup( 'UGC', '#FFFF00'),

ColorGroup( 'TAA', '#000000'),
ColorGroup( 'UAA', '#000000'),
ColorGroup( 'TAG', '#000000'),
ColorGroup( 'UAG', '#000000'),
ColorGroup( 'TGA', '#000000'),  
ColorGroup( 'UGA', '#000000')],
  alphabet = codon_alphabetU
  )
codonsT = ColorScheme([
ColorGroup( 'CAT', '#00FFFF'),
ColorGroup( 'CAU', '#00FFFF'),
ColorGroup( 'CAC', '#00FFFF'),

ColorGroup( 'AAA', '#00FFFF'),
ColorGroup( 'AAG', '#00FFFF'),

ColorGroup( 'CGT', '#00FFFF'),
ColorGroup( 'CGU', '#00FFFF'),
ColorGroup( 'CGC', '#00FFFF'),
ColorGroup( 'CGA', '#00FFFF'),
ColorGroup( 'CGG', '#00FFFF'),
ColorGroup( 'AGA', '#00FFFF'),
ColorGroup( 'AGG', '#00FFFF'),


ColorGroup( 'GAT', '#FF0000'),
ColorGroup( 'GAU', '#FF0000'),
ColorGroup( 'GAC', '#FF0000'),

ColorGroup( 'GAA', '#FF0000'),
ColorGroup( 'GAG', '#FF0000'),


ColorGroup( 'TCT', '#00FF00'),
ColorGroup( 'UCU', '#00FF00'),
ColorGroup( 'TCC', '#00FF00'),
ColorGroup( 'UCC', '#00FF00'),
ColorGroup( 'TCA', '#00FF00'),
ColorGroup( 'UCA', '#00FF00'),
ColorGroup( 'TCG', '#00FF00'),
ColorGroup( 'UCG', '#00FF00'),
ColorGroup( 'AGT', '#00FF00'),
ColorGroup( 'AGU', '#00FF00'),
ColorGroup( 'AGC', '#00FF00'),

ColorGroup( 'ACT', '#00FF00'),
ColorGroup( 'ACU', '#00FF00'),
ColorGroup( 'ACC', '#00FF00'),
ColorGroup( 'ACA', '#00FF00'),
ColorGroup( 'ACG', '#00FF00'),

ColorGroup( 'CAA', '#00FF00'),
ColorGroup( 'CAG', '#00FF00'),

ColorGroup( 'AAT', '#00FF00'),
ColorGroup( 'AAU', '#00FF00'),
ColorGroup( 'AAC', '#00FF00'),


ColorGroup( 'GCT', '#5555FF'),
ColorGroup( 'GCU', '#5555FF'),
ColorGroup( 'GCC', '#5555FF'),
ColorGroup( 'GCA', '#5555FF'),
ColorGroup( 'GCG', '#5555FF'),

ColorGroup( 'GTT', '#5555FF'),
ColorGroup( 'GUU', '#5555FF'),
ColorGroup( 'GTC', '#5555FF'),
ColorGroup( 'GUC', '#5555FF'),
ColorGroup( 'GTA', '#5555FF'),
ColorGroup( 'GUA', '#5555FF'),
ColorGroup( 'GTG', '#5555FF'),
ColorGroup( 'GUG', '#5555FF'),

ColorGroup( 'CTT', '#5555FF'),
ColorGroup( 'CUU', '#5555FF'),
ColorGroup( 'CTC', '#5555FF'),
ColorGroup( 'CUC', '#5555FF'),
ColorGroup( 'CTA', '#5555FF'),
ColorGroup( 'CUA', '#5555FF'),
ColorGroup( 'CTG', '#5555FF'),
ColorGroup( 'CUG', '#5555FF'),
ColorGroup( 'TTA', '#5555FF'),
ColorGroup( 'UUA', '#5555FF'),
ColorGroup( 'TTG', '#5555FF'),
ColorGroup( 'UUG', '#5555FF'),

ColorGroup( 'ATT', '#5555FF'),
ColorGroup( 'AUU', '#5555FF'),
ColorGroup( 'ATC', '#5555FF'),
ColorGroup( 'AUC', '#5555FF'),
ColorGroup( 'ATA', '#5555FF'),
ColorGroup( 'AUA', '#5555FF'),

ColorGroup( 'ATG', '#5555FF'),
ColorGroup( 'AUG', '#5555FF'),


ColorGroup( 'TTT', '#FF00FF'),
ColorGroup( 'UUU', '#FF00FF'),
ColorGroup( 'TTC', '#FF00FF'),
ColorGroup( 'UUC', '#FF00FF'),

ColorGroup( 'TAT', '#FF00FF'),
ColorGroup( 'UAU', '#FF00FF'),
ColorGroup( 'TAC', '#FF00FF'),
ColorGroup( 'UAC', '#FF00FF'),

ColorGroup( 'TGG', '#FF00FF'),
ColorGroup( 'UGG', '#FF00FF'),


ColorGroup( 'GGT', '#996600'),
ColorGroup( 'GGU', '#996600'),
ColorGroup( 'GGC', '#996600'),
ColorGroup( 'GGA', '#996600'),
ColorGroup( 'GGG', '#996600'),

ColorGroup( 'CCT', '#996600'),
ColorGroup( 'CCU', '#996600'),
ColorGroup( 'CCC', '#996600'),
ColorGroup( 'CCA', '#996600'),
ColorGroup( 'CCG', '#996600'),


ColorGroup( 'TGT', '#FFFF00'),
ColorGroup( 'UGU', '#FFFF00'),
ColorGroup( 'TGC', '#FFFF00'),
ColorGroup( 'UGC', '#FFFF00'),

ColorGroup( 'TAA', '#000000'),
ColorGroup( 'UAA', '#000000'),
ColorGroup( 'TAG', '#000000'),
ColorGroup( 'UAG', '#000000'),
ColorGroup( 'TGA', '#000000'),  
ColorGroup( 'UGA', '#000000')],
  alphabet = codon_alphabetT
  )




charge = ColorScheme([
    ColorGroup("KRH", "blue", "Positive" ),
    ColorGroup( "DE", "red", "Negative") ],
    alphabet = seq.unambiguous_protein_alphabet
    )


taylor = ColorScheme([
    ColorGroup( 'A', '#CCFF00' ),
    ColorGroup( 'C', '#FFFF00' ),
    ColorGroup( 'D', '#FF0000'),
    ColorGroup( 'E', '#FF0066' ),
    ColorGroup( 'F', '#00FF66'),
    ColorGroup( 'G', '#FF9900'),
    ColorGroup( 'H', '#0066FF'),
    ColorGroup( 'I', '#66FF00'),
    ColorGroup( 'K', '#6600FF'),
    ColorGroup( 'L', '#33FF00'),
    ColorGroup( 'M', '#00FF00'),
    ColorGroup( 'N', '#CC00FF'),
    ColorGroup( 'P', '#FFCC00'),
    ColorGroup( 'Q', '#FF00CC'),
    ColorGroup( 'R', '#0000FF'),
    ColorGroup( 'S', '#FF3300'),
    ColorGroup( 'T', '#FF6600'),
    ColorGroup( 'V', '#99FF00'),
    ColorGroup( 'W', '#00CCFF'),
    ColorGroup( 'Y', '#00FFCC')],
    title = "Taylor",
    description = "W. Taylor, Protein Engineering, Vol 10 , 743-746 (1997)",
    alphabet = seq.unambiguous_protein_alphabet
    )
    


