package datastructures;

/**
 * Used for various purposes.
 * 
 * @author Ali Abdallah
 * @version 01.2011
 * @since jdk 1.6.0
 */
public class Frame {
	
	// Self explanatory.
	protected int start; 
	protected int end;
	protected int length;
	
	/**
	 * Constructs a frame with start position "start", length "length" and
	 * end position "start+length-1".
	 * @param start the start position of the frame.
	 * @param length the length of the frame.
	 */
	public Frame(int start, int length) {
		this.start = start; end = start+length-1;
		this.length = length;
	}
	
	/**
	 * Verify if this frame overlaps the frame specified in the parameter.
	 * @param f the frame which potentially overlaps the calling frame.
	 * @return true if the two frames overlap each other and false otherwise.
	 */
	public boolean overlaps(Frame f){
		return (end >= f.start() && start < f.end());
	}
	
	/**
	 * Computes the size of the overlap area.
	 * @param f the frame overlapping the calling frame.
	 * @return the size of the overlap if an overlap exists and -1 otherwise.
	 */
	public int overlapSize(Frame f){
		if(this.overlaps(f)){
			int s = Math.max(start, f.start());
			int e = Math.min(end, f.end());
			return e-s+1;
		}
		return -1;
	}
	
	/**
	 * Unify the calling frame with the frame specified in the parameter.
	 * @param f the frame to be unified with the calling frame.
	 * @return the union of the calling frame and f.
	 */
	public Frame unify(Frame f){
		return 
			new Frame(Math.min(this.start, f.start), 
					  Math.max(this.end,f.end)-Math.min(this.start, f.start)+1);	
	}

	/**
	 * self explanatory
	 * @return start of the frame.
	 */
	public int start(){
		return start;
	}
	
	/**
	 * self explanatory
	 * @return end of the frame.
	 */
	public int end(){
		return end;
	}
	
	/**
	 * self explanatory
	 * @return length of the frame.
	 */
	public int length(){
		return length;
	}
	
	/**
	 * Make a string representation of the frame.
	 * @return a string representation of the string.
	 */
	public String toString(){
		return start+":"+end;
	}

}
