/* 
 * File:   notes.h
 * Author: rgoya
 *
 * Created on February 3, 2010, 12:37 PM
 */

/*! @typedef
@abstract Structure for one alignment covering the pileup position.
@field  b      pointer to the alignment
@field  qpos   position of the read base at the pileup site, 0-based
@field  indel  indel length; 0 for no indel, positive for ins and negative for del
@field  is_del 1 iff the base on the padded read is a deletion
@field  level  the level of the read in the "viewer" mode

@discussion See also bam_plbuf_push() and bam_lplbuf_push(). The
difference between the two functions is that the former does not
set bam_pileup1_t::level, while the later does. Level helps the
implementation of alignment viewers, but calculating this has some
overhead.
*/
typedef struct {
	bam1_t *b;
	int32_t qpos;
	int indel, level;
	uint32_t is_del:1, is_head:1, is_tail:1;
} bam_pileup1_t;

/*! @typedef
  @abstract Structure for core alignment information.
  @field  tid     chromosome ID, defined by bam_header_t
  @field  pos     0-based leftmost coordinate
  @field  strand  strand; 0 for forward and 1 otherwise
  @field  bin     bin calculated by bam_reg2bin()
  @field  qual    mapping quality
  @field  l_qname length of the query name
  @field  flag    bitwise flag
  @field  n_cigar number of CIGAR operations
  @field  l_qseq  length of the query sequence (read)
 */
typedef struct {
	int32_t tid;
	int32_t pos;
	uint32_t bin:16, qual:8, l_qname:8;
	uint32_t flag:16, n_cigar:16;
	int32_t l_qseq;
	int32_t mtid;
	int32_t mpos;
	int32_t isize;
} bam1_core_t;

/*! @typedef
  @abstract Structure for one alignment.
  @field  core       core information about the alignment
  @field  l_aux      length of auxiliary data
  @field  data_len   current length of bam1_t::data
  @field  m_data     maximum length of bam1_t::data
  @field  data       all variable-length data, concatenated; structure: cigar-qname-seq-qual-aux

  @discussion Notes:

   1. qname is zero tailing and core.l_qname includes the tailing '\0'.
   2. l_qseq is calculated from the total length of an alignment block
      on reading or from CIGAR.
 */
typedef struct {
	bam1_core_t core;
	int l_aux, data_len, m_data;
	uint8_t *data;
} bam1_t;


/*! @typedef
@abstract Structure for one alignment covering the pileup position.
@field  b      pointer to the alignment
@field  qpos   position of the read base at the pileup site, 0-based
@field  indel  indel length; 0 for no indel, positive for ins and negative for del
@field  is_del 1 iff the base on the padded read is a deletion
@field  level  the level of the read in the "viewer" mode

@discussion See also bam_plbuf_push() and bam_lplbuf_push(). The
difference between the two functions is that the former does not
set bam_pileup1_t::level, while the later does. Level helps the
implementation of alignment viewers, but calculating this has some
overhead.
*/
typedef struct {
	bam1_t *b;
	int32_t qpos;
	int indel, level;
	uint32_t is_del:1, is_head:1, is_tail:1;
} bam_pileup1_t;
