#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <math.h>
#include "sam_header.h"
#include "sam.h"
#include "faidx.h"
#include "kstring.h"
#include "khash.h"
KHASH_SET_INIT_STR(rg)

// When counting records instead of printing them,
// data passed to the bam_fetch callback is encapsulated in this struct.
typedef struct {
	bam_header_t *header;
	int *count;
} count_func_data_t;

typedef khash_t(rg) *rghash_t;

static rghash_t g_rghash = 0;
static int g_min_mapQ = 0, g_flag_on = 0, g_flag_off = 0;
static char *g_library, *g_rg;
static int g_sol2sanger_tbl[128];
static void *g_bed;

void *bed_read(const char *fn);
void bed_destroy(void *_h);
int bed_overlap(const void *_h, const char *chr, int beg, int end);

static void sol2sanger(bam1_t *b)
{
	int l;
	uint8_t *qual = bam1_qual(b);
	if (g_sol2sanger_tbl[30] == 0) {
		for (l = 0; l != 128; ++l) {
			g_sol2sanger_tbl[l] = (int)(10.0 * log(1.0 + pow(10.0, (l - 64 + 33) / 10.0)) / log(10.0) + .499);
			if (g_sol2sanger_tbl[l] >= 93) g_sol2sanger_tbl[l] = 93;
		}
	}
	for (l = 0; l < b->core.l_qseq; ++l) {
		int q = qual[l];
		if (q > 127) q = 127;
		qual[l] = g_sol2sanger_tbl[q];
	}
}

static inline int __g_skip_aln(const bam_header_t *h, const bam1_t *b)
{
	if (b->core.qual < g_min_mapQ || ((b->core.flag & g_flag_on) != g_flag_on) || (b->core.flag & g_flag_off))
		return 1;
	if (g_bed && b->core.tid >= 0 && !bed_overlap(g_bed, h->target_name[b->core.tid], b->core.pos, bam_calend(&b->core, bam1_cigar(b))))
		return 1;
	if (g_rg || g_rghash) {
		uint8_t *s = bam_aux_get(b, "RG");
		if (s) {
			if (g_rg) return (strcmp(g_rg, (char*)(s + 1)) == 0)? 0 : 1;
			if (g_rghash) {
				khint_t k = kh_get(rg, g_rghash, (char*)(s + 1));
				return (k != kh_end(g_rghash))? 0 : 1;
			}
		}
	}
	if (g_library) {
		const char *p = bam_get_library((bam_header_t*)h, b);
		return (p && strcmp(p, g_library) == 0)? 0 : 1;
	}
	return 0;
}

static char *drop_rg(char *hdtxt, rghash_t h, int *len)
{
	char *p = hdtxt, *q, *r, *s;
	kstring_t str;
	memset(&str, 0, sizeof(kstring_t));
	while (1) {
		int toprint = 0;
		q = strchr(p, '\n');
		if (q == 0) q = p + strlen(p);
		if (q - p < 3) break; // the line is too short; then stop
		if (strncmp(p, "@RG\t", 4) == 0) {
			int c;
			khint_t k;
			if ((r = strstr(p, "\tID:")) != 0) {
				r += 4;
				for (s = r; *s != '\0' && *s != '\n' && *s != '\t'; ++s);
				c = *s; *s = '\0';
				k = kh_get(rg, h, r);
				*s = c;
				if (k != kh_end(h)) toprint = 1;
			}
		} else toprint = 1;
		if (toprint) {
			kputsn(p, q - p, &str); kputc('\n', &str);
		}
		p = q + 1;
	}
	*len = str.l;
	return str.s;
}

// callback function for bam_fetch() that prints nonskipped records
static int view_func(const bam1_t *b, void *data)
{
	if (!__g_skip_aln(((samfile_t*)data)->header, b))
		samwrite((samfile_t*)data, b);
	return 0;
}

// callback function for bam_fetch() that counts nonskipped records
static int count_func(const bam1_t *b, void *data)
{
	if (!__g_skip_aln(((count_func_data_t*)data)->header, b)) {
		(*((count_func_data_t*)data)->count)++;
	}
	return 0;
}

static int usage(int is_long_help);

int main_samview(int argc, char *argv[])
{
	int c, is_header = 0, is_header_only = 0, is_bamin = 1, ret = 0, compress_level = -1, is_bamout = 0, slx2sngr = 0, is_count = 0;
	int of_type = BAM_OFDEC, is_long_help = 0;
	int count = 0;
	samfile_t *in = 0, *out = 0;
	char in_mode[5], out_mode[5], *fn_out = 0, *fn_list = 0, *fn_ref = 0, *fn_rg = 0;

	/* parse command-line options */
	strcpy(in_mode, "r"); strcpy(out_mode, "w");
	while ((c = getopt(argc, argv, "Sbct:h1Ho:q:f:F:ul:r:xX?T:CR:L:")) >= 0) {
		switch (c) {
		case 'c': is_count = 1; break;
		case 'C': slx2sngr = 1; break;
		case 'S': is_bamin = 0; break;
		case 'b': is_bamout = 1; break;
		case 't': fn_list = strdup(optarg); is_bamin = 0; break;
		case 'h': is_header = 1; break;
		case 'H': is_header_only = 1; break;
		case 'o': fn_out = strdup(optarg); break;
		case 'f': g_flag_on = strtol(optarg, 0, 0); break;
		case 'F': g_flag_off = strtol(optarg, 0, 0); break;
		case 'q': g_min_mapQ = atoi(optarg); break;
		case 'u': compress_level = 0; break;
		case '1': compress_level = 1; break;
		case 'l': g_library = strdup(optarg); break;
		case 'L': g_bed = bed_read(optarg); break;
		case 'r': g_rg = strdup(optarg); break;
		case 'R': fn_rg = strdup(optarg); break;
		case 'x': of_type = BAM_OFHEX; break;
		case 'X': of_type = BAM_OFSTR; break;
		case '?': is_long_help = 1; break;
		case 'T': fn_ref = strdup(optarg); is_bamin = 0; break;
		default: return usage(is_long_help);
		}
	}
	if (compress_level >= 0) is_bamout = 1;
	if (is_header_only) is_header = 1;
	if (is_bamout) strcat(out_mode, "b");
	else {
		if (of_type == BAM_OFHEX) strcat(out_mode, "x");
		else if (of_type == BAM_OFSTR) strcat(out_mode, "X");
	}
	if (is_bamin) strcat(in_mode, "b");
	if (is_header) strcat(out_mode, "h");
	if (compress_level >= 0) {
		char tmp[2];
		tmp[0] = compress_level + '0'; tmp[1] = '\0';
		strcat(out_mode, tmp);
	}
	if (argc == optind) return usage(is_long_help); // potential memory leak...

	// read the list of read groups
	if (fn_rg) {
		FILE *fp_rg;
		char buf[1024];
		int ret;
		g_rghash = kh_init(rg);
		fp_rg = fopen(fn_rg, "r");
		while (!feof(fp_rg) && fscanf(fp_rg, "%s", buf) > 0) // this is not a good style, but bear me...
			kh_put(rg, g_rghash, strdup(buf), &ret); // we'd better check duplicates...
		fclose(fp_rg);
	}

	// generate the fn_list if necessary
	if (fn_list == 0 && fn_ref) fn_list = samfaipath(fn_ref);
	// open file handlers
	if ((in = samopen(argv[optind], in_mode, fn_list)) == 0) {
		fprintf(stderr, "[main_samview] fail to open \"%s\" for reading.\n", argv[optind]);
		ret = 1;
		goto view_end;
	}
	if (in->header == 0) {
		fprintf(stderr, "[main_samview] fail to read the header from \"%s\".\n", argv[optind]);
		ret = 1;
		goto view_end;
	}
	if (g_rghash) { // FIXME: I do not know what "bam_header_t::n_text" is for...
		char *tmp;
		int l;
		tmp = drop_rg(in->header->text, g_rghash, &l);
		free(in->header->text);
		in->header->text = tmp;
		in->header->l_text = l;
	}
	if (!is_count && (out = samopen(fn_out? fn_out : "-", out_mode, in->header)) == 0) {
		fprintf(stderr, "[main_samview] fail to open \"%s\" for writing.\n", fn_out? fn_out : "standard output");
		ret = 1;
		goto view_end;
	}
	if (is_header_only) goto view_end; // no need to print alignments

	if (argc == optind + 1) { // convert/print the entire file
		bam1_t *b = bam_init1();
		int r;
		while ((r = samread(in, b)) >= 0) { // read one alignment from `in'
			if (!__g_skip_aln(in->header, b)) {
				if (slx2sngr) sol2sanger(b);
				if (!is_count) samwrite(out, b); // write the alignment to `out'
				count++;
			}
		}
		if (r < -1) {
			fprintf(stderr, "[main_samview] truncated file.\n");
			ret = 1;
		}
		bam_destroy1(b);
	} else { // retrieve alignments in specified regions
		int i;
		bam_index_t *idx = 0;
		if (is_bamin) idx = bam_index_load(argv[optind]); // load BAM index
		if (idx == 0) { // index is unavailable
			fprintf(stderr, "[main_samview] random alignment retrieval only works for indexed BAM files.\n");
			ret = 1;
			goto view_end;
		}
		for (i = optind + 1; i < argc; ++i) {
			int tid, beg, end, result;
			bam_parse_region(in->header, argv[i], &tid, &beg, &end); // parse a region in the format like `chr2:100-200'
			if (tid < 0) { // reference name is not found
				fprintf(stderr, "[main_samview] region \"%s\" specifies an unknown reference name. Continue anyway.\n", argv[i]);
				continue;
			}
			// fetch alignments
			if (is_count) {
				count_func_data_t count_data = { in->header, &count };
				result = bam_fetch(in->x.bam, idx, tid, beg, end, &count_data, count_func);
			} else
				result = bam_fetch(in->x.bam, idx, tid, beg, end, out, view_func);
			if (result < 0) {
				fprintf(stderr, "[main_samview] retrieval of region \"%s\" failed due to truncated file or corrupt BAM index file\n", argv[i]);
				ret = 1;
				break;
			}
		}
		bam_index_destroy(idx); // destroy the BAM index
	}

view_end:
	if (is_count && ret == 0) {
		printf("%d\n", count);
	}
	// close files, free and return
	free(fn_list); free(fn_ref); free(fn_out); free(g_library); free(g_rg); free(fn_rg);
	if (g_bed) bed_destroy(g_bed);
	if (g_rghash) {
		khint_t k;
		for (k = 0; k < kh_end(g_rghash); ++k)
			if (kh_exist(g_rghash, k)) free((char*)kh_key(g_rghash, k));
		kh_destroy(rg, g_rghash);
	}
	samclose(in);
	if (!is_count)
		samclose(out);
	return ret;
}

static int usage(int is_long_help)
{
	fprintf(stderr, "\n");
	fprintf(stderr, "Usage:   samtools view [options] <in.bam>|<in.sam> [region1 [...]]\n\n");
	fprintf(stderr, "Options: -b       output BAM\n");
	fprintf(stderr, "         -h       print header for the SAM output\n");
	fprintf(stderr, "         -H       print header only (no alignments)\n");
	fprintf(stderr, "         -S       input is SAM\n");
	fprintf(stderr, "         -u       uncompressed BAM output (force -b)\n");
	fprintf(stderr, "         -1       fast compression (force -b)\n");
	fprintf(stderr, "         -x       output FLAG in HEX (samtools-C specific)\n");
	fprintf(stderr, "         -X       output FLAG in string (samtools-C specific)\n");
	fprintf(stderr, "         -c       print only the count of matching records\n");
	fprintf(stderr, "         -L FILE  output alignments overlapping the input BED FILE [null]\n");
	fprintf(stderr, "         -t FILE  list of reference names and lengths (force -S) [null]\n");
	fprintf(stderr, "         -T FILE  reference sequence file (force -S) [null]\n");
	fprintf(stderr, "         -o FILE  output file name [stdout]\n");
	fprintf(stderr, "         -R FILE  list of read groups to be outputted [null]\n");
	fprintf(stderr, "         -f INT   required flag, 0 for unset [0]\n");
	fprintf(stderr, "         -F INT   filtering flag, 0 for unset [0]\n");
	fprintf(stderr, "         -q INT   minimum mapping quality [0]\n");
	fprintf(stderr, "         -l STR   only output reads in library STR [null]\n");
	fprintf(stderr, "         -r STR   only output reads in read group STR [null]\n");
	fprintf(stderr, "         -?       longer help\n");
	fprintf(stderr, "\n");
	if (is_long_help)
		fprintf(stderr, "Notes:\n\
\n\
  1. By default, this command assumes the file on the command line is in\n\
     the BAM format and it prints the alignments in SAM. If `-t' is\n\
     applied, the input file is assumed to be in the SAM format. The\n\
     file supplied with `-t' is SPACE/TAB delimited with the first two\n\
     fields of each line consisting of the reference name and the\n\
     corresponding sequence length. The `.fai' file generated by `faidx'\n\
     can be used here. This file may be empty if reads are unaligned.\n\
\n\
  2. SAM->BAM conversion: `samtools view -bT ref.fa in.sam.gz'.\n\
\n\
  3. BAM->SAM conversion: `samtools view in.bam'.\n\
\n\
  4. A region should be presented in one of the following formats:\n\
     `chr1', `chr2:1,000' and `chr3:1000-2,000'. When a region is\n\
     specified, the input alignment file must be an indexed BAM file.\n\
\n\
  5. Option `-u' is preferred over `-b' when the output is piped to\n\
     another samtools command.\n\
\n\
  6. In a string FLAG, each character represents one bit with\n\
     p=0x1 (paired), P=0x2 (properly paired), u=0x4 (unmapped),\n\
     U=0x8 (mate unmapped), r=0x10 (reverse), R=0x20 (mate reverse)\n\
     1=0x40 (first), 2=0x80 (second), s=0x100 (not primary), \n\
     f=0x200 (failure) and d=0x400 (duplicate). Note that `-x' and\n\
     `-X' are samtools-C specific. Picard and older samtools do not\n\
     support HEX or string flags.\n\
\n");
	return 1;
}

int main_import(int argc, char *argv[])
{
	int argc2, ret;
	char **argv2;
	if (argc != 4) {
		fprintf(stderr, "Usage: bamtk import <in.ref_list> <in.sam> <out.bam>\n");
		return 1;
	}
	argc2 = 6;
	argv2 = calloc(6, sizeof(char*));
	argv2[0] = "import", argv2[1] = "-o", argv2[2] = argv[3], argv2[3] = "-bt", argv2[4] = argv[1], argv2[5] = argv[2];
	ret = main_samview(argc2, argv2);
	free(argv2);
	return ret;
}
