/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.fasta;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Arrays;
import net.sf.picard.PicardException;
import net.sf.picard.reference.FastaSequenceIndex;
import net.sf.picard.reference.IndexedFastaSequenceFile;
import net.sf.picard.reference.ReferenceSequence;
import net.sf.samtools.SAMSequenceRecord;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;

public class CachingIndexedFastaSequenceFile
extends IndexedFastaSequenceFile {
    private static final boolean USE_CACHE = true;
    private static final boolean PRINT_EFFICIENCY = false;
    private static final int PRINT_FREQUENCY = 10000;
    private static final long DEFAULT_CACHE_SIZE = 1000000L;
    long cacheHits = 0L;
    long cacheMisses = 0L;
    long cacheSize = 1000000L;
    long cacheMissBackup = 100L;
    private ThreadLocal<Cache> cache;

    protected void resetThreadLocalCache() {
        this.cache = new ThreadLocal<Cache>(){

            @Override
            protected Cache initialValue() {
                return new Cache();
            }
        };
    }

    public CachingIndexedFastaSequenceFile(File file, FastaSequenceIndex index, long cacheSize) {
        super(file, index);
        this.resetThreadLocalCache();
        this.setCacheSize(cacheSize);
    }

    private void setCacheSize(long cacheSize) {
        this.cacheSize = cacheSize;
        this.cacheMissBackup = Math.max(cacheSize / 1000L, 1L);
    }

    public CachingIndexedFastaSequenceFile(File file, FastaSequenceIndex index) {
        this(file, index, 1000000L);
    }

    public CachingIndexedFastaSequenceFile(File file) throws FileNotFoundException {
        this(file, 1000000L);
    }

    public CachingIndexedFastaSequenceFile(File file, long cacheSize) throws FileNotFoundException {
        super(file);
        this.resetThreadLocalCache();
        this.setCacheSize(cacheSize);
    }

    public void printEfficiency() {
        if ((this.cacheHits + this.cacheMisses) % 10000L == 0L) {
            System.out.printf("### CachingIndexedFastaReader: hits=%d misses=%d efficiency %.6f%%%n", this.cacheHits, this.cacheMisses, this.calcEfficiency());
        }
    }

    public double calcEfficiency() {
        return 100.0 * (double)this.cacheHits / ((double)this.cacheMisses + (double)this.cacheHits * 1.0);
    }

    public long getCacheHits() {
        return this.cacheHits;
    }

    public long getCacheMisses() {
        return this.cacheMisses;
    }

    @Override
    public ReferenceSequence getSubsequenceAt(String contig, long start, long stop) {
        ReferenceSequence result;
        Cache myCache = this.cache.get();
        if (stop - start >= this.cacheSize) {
            ++this.cacheMisses;
            result = super.getSubsequenceAt(contig, start, stop);
        } else {
            SAMSequenceRecord contigInfo = super.getSequenceDictionary().getSequence(contig);
            if (stop > (long)contigInfo.getSequenceLength()) {
                throw new PicardException("Query asks for data past end of contig");
            }
            if (start < myCache.start || stop > myCache.stop || myCache.seq == null || myCache.seq.getContigIndex() != contigInfo.getSequenceIndex()) {
                ++this.cacheMisses;
                myCache.start = Math.max(start - this.cacheMissBackup, 0L);
                myCache.stop = Math.min(myCache.start + this.cacheSize, (long)contigInfo.getSequenceLength());
                myCache.seq = super.getSubsequenceAt(contig, myCache.start, myCache.stop);
            } else {
                ++this.cacheHits;
            }
            int cacheOffsetStart = (int)(start - myCache.start);
            int cacheOffsetStop = (int)(stop - start + (long)cacheOffsetStart + 1L);
            try {
                result = new ReferenceSequence(myCache.seq.getName(), myCache.seq.getContigIndex(), Arrays.copyOfRange(myCache.seq.getBases(), cacheOffsetStart, cacheOffsetStop));
            }
            catch (ArrayIndexOutOfBoundsException e) {
                throw new ReviewedStingException(String.format("BUG: bad array indexing.  Cache start %d and end %d, request start %d end %d, offset start %d and end %d, base size %d", myCache.start, myCache.stop, start, stop, cacheOffsetStart, cacheOffsetStop, myCache.seq.getBases().length), e);
            }
        }
        return result;
    }

    private static class Cache {
        long start = -1L;
        long stop = -1L;
        ReferenceSequence seq = null;

        private Cache() {
        }
    }
}

