/*
 * Decompiled with CFR 0.152.
 */
package net.sf.samtools;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.List;
import net.sf.samtools.BAMFileConstants;
import net.sf.samtools.BAMIndex;
import net.sf.samtools.BAMIndexContent;
import net.sf.samtools.BAMIndexMetaData;
import net.sf.samtools.Bin;
import net.sf.samtools.Chunk;
import net.sf.samtools.LinearIndex;
import net.sf.samtools.SAMException;
import net.sf.samtools.SAMSequenceDictionary;
import net.sf.samtools.util.RuntimeIOException;

public abstract class AbstractBAMFileIndex
implements BAMIndex {
    protected static final int BIN_GENOMIC_SPAN = 0x20000000;
    private static final int[] LEVEL_STARTS = new int[]{0, 1, 9, 73, 585, 4681};
    public static final int MAX_BINS = 37450;
    public static final int MAX_LINEAR_INDEX_SIZE = 37451 - LEVEL_STARTS[LEVEL_STARTS.length - 1];
    private final File mFile;
    private final MappedByteBuffer mFileBuffer;
    private SAMSequenceDictionary mBamDictionary = null;

    protected AbstractBAMFileIndex(File file, SAMSequenceDictionary dictionary) {
        this.mFile = file;
        this.mBamDictionary = dictionary;
        try {
            FileInputStream fileStream = new FileInputStream(this.mFile);
            FileChannel fileChannel = fileStream.getChannel();
            this.mFileBuffer = fileChannel.map(FileChannel.MapMode.READ_ONLY, 0L, fileChannel.size());
            this.mFileBuffer.order(ByteOrder.LITTLE_ENDIAN);
            fileChannel.close();
            fileStream.close();
        }
        catch (IOException exc) {
            throw new RuntimeIOException(exc.getMessage(), exc);
        }
        this.seek(0);
        byte[] buffer = new byte[4];
        this.readBytes(buffer);
        if (!Arrays.equals(buffer, BAMFileConstants.BAM_INDEX_MAGIC)) {
            throw new RuntimeException("Invalid file header in BAM index " + this.mFile + ": " + new String(buffer));
        }
    }

    public static int getNumIndexLevels() {
        return LEVEL_STARTS.length;
    }

    public static int getFirstBinInLevel(int levelNumber) {
        return LEVEL_STARTS[levelNumber];
    }

    public int getLevelSize(int levelNumber) {
        if (levelNumber == AbstractBAMFileIndex.getNumIndexLevels()) {
            return 37451 - LEVEL_STARTS[levelNumber];
        }
        return LEVEL_STARTS[levelNumber + 1] - LEVEL_STARTS[levelNumber];
    }

    public int getLevelForBin(Bin bin) {
        if (bin.getBinNumber() >= 37450) {
            throw new SAMException("Tried to get level for invalid bin.");
        }
        for (int i = AbstractBAMFileIndex.getNumIndexLevels() - 1; i >= 0; --i) {
            if (bin.getBinNumber() < LEVEL_STARTS[i]) continue;
            return i;
        }
        throw new SAMException("Unable to find correct bin for bin " + bin);
    }

    public int getFirstLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == AbstractBAMFileIndex.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (bin.getBinNumber() - levelStart) * (0x20000000 / levelSize) + 1;
    }

    public int getLastLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == AbstractBAMFileIndex.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (bin.getBinNumber() - levelStart + 1) * (0x20000000 / levelSize);
    }

    public int getNumberOfReferences() {
        this.seek(4);
        return this.readInteger();
    }

    @Override
    public long getStartOfLastLinearBin() {
        this.seek(4);
        int sequenceCount = this.readInteger();
        long lastLinearIndexPointer = -1L;
        for (int i = 0; i < sequenceCount; ++i) {
            int nBins = this.readInteger();
            for (int j1 = 0; j1 < nBins; ++j1) {
                this.skipBytes(4);
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            if (nLinearBins <= 0) continue;
            this.skipBytes(8 * (nLinearBins - 1));
            lastLinearIndexPointer = this.readLong();
        }
        return lastLinearIndexPointer;
    }

    Long getNoCoordinateCount() {
        try {
            return this.readLong();
        }
        catch (Exception e) {
            return null;
        }
    }

    protected BAMIndexContent query(int referenceSequence, int startPos, int endPos) {
        this.seek(4);
        Bin[] bins = null;
        LinearIndex linearIndex = null;
        ArrayList<Chunk> metaDataChunks = new ArrayList<Chunk>();
        int sequenceCount = this.readInteger();
        if (referenceSequence >= sequenceCount) {
            return null;
        }
        BitSet regionBins = this.regionToBins(startPos, endPos);
        if (regionBins == null) {
            return null;
        }
        this.skipToSequence(referenceSequence);
        int binCount = this.readInteger();
        boolean metaDataSeen = false;
        bins = new Bin[this.getMaxBinNumberForReference(referenceSequence) + 1];
        for (int binNumber = 0; binNumber < binCount; ++binNumber) {
            long chunkEnd;
            long chunkBegin;
            int indexBin = this.readInteger();
            int nChunks = this.readInteger();
            ArrayList<Chunk> chunks = new ArrayList<Chunk>(nChunks);
            Chunk lastChunk = null;
            if (regionBins.get(indexBin)) {
                for (int ci = 0; ci < nChunks; ++ci) {
                    chunkBegin = this.readLong();
                    chunkEnd = this.readLong();
                    lastChunk = new Chunk(chunkBegin, chunkEnd);
                    chunks.add(lastChunk);
                }
            } else {
                if (indexBin == 37450) {
                    for (int ci = 0; ci < nChunks; ++ci) {
                        chunkBegin = this.readLong();
                        chunkEnd = this.readLong();
                        lastChunk = new Chunk(chunkBegin, chunkEnd);
                        metaDataChunks.add(lastChunk);
                    }
                    metaDataSeen = true;
                    continue;
                }
                this.skipBytes(16 * nChunks);
            }
            Bin bin = new Bin(referenceSequence, indexBin);
            bin.setChunkList(chunks);
            bin.setLastChunk(lastChunk);
            bins[bin.getBinNumber()] = bin;
        }
        int nLinearBins = this.readInteger();
        int regionLinearBinStart = LinearIndex.convertToLinearIndexOffset(startPos);
        int regionLinearBinStop = LinearIndex.convertToLinearIndexOffset(endPos) > 0 ? LinearIndex.convertToLinearIndexOffset(endPos) : nLinearBins - 1;
        int actualStop = Math.min(regionLinearBinStop, nLinearBins - 1);
        long[] linearIndexEntries = new long[]{};
        if (regionLinearBinStart < nLinearBins) {
            linearIndexEntries = new long[actualStop - regionLinearBinStart + 1];
            this.skipBytes(8 * regionLinearBinStart);
            for (int linearBin = regionLinearBinStart; linearBin <= actualStop; ++linearBin) {
                linearIndexEntries[linearBin - regionLinearBinStart] = this.readLong();
            }
        }
        linearIndex = new LinearIndex(referenceSequence, regionLinearBinStart, linearIndexEntries);
        return new BAMIndexContent(referenceSequence, bins, binCount - (metaDataSeen ? 1 : 0), new BAMIndexMetaData(metaDataChunks), linearIndex);
    }

    private int getMaxBinNumberForReference(int reference) {
        try {
            int sequenceLength = this.mBamDictionary.getSequence(reference).getSequenceLength();
            return AbstractBAMFileIndex.getMaxBinNumberForSequenceLength(sequenceLength);
        }
        catch (Exception e) {
            return 37450;
        }
    }

    static int getMaxBinNumberForSequenceLength(int sequenceLength) {
        return AbstractBAMFileIndex.getFirstBinInLevel(AbstractBAMFileIndex.getNumIndexLevels() - 1) + (sequenceLength >> 14);
    }

    protected abstract BAMIndexContent getQueryResults(int var1);

    protected int getMaxAddressibleGenomicLocation() {
        return 0x20000000;
    }

    protected BitSet regionToBins(int startPos, int endPos) {
        int k;
        int end;
        int maxPos = 0x1FFFFFFF;
        int start = startPos <= 0 ? 0 : startPos - 1 & 0x1FFFFFFF;
        int n = end = endPos <= 0 ? 0x1FFFFFFF : endPos - 1 & 0x1FFFFFFF;
        if (start > end) {
            return null;
        }
        BitSet bitSet = new BitSet(37450);
        bitSet.set(0);
        for (k = 1 + (start >> 26); k <= 1 + (end >> 26); ++k) {
            bitSet.set(k);
        }
        for (k = 9 + (start >> 23); k <= 9 + (end >> 23); ++k) {
            bitSet.set(k);
        }
        for (k = 73 + (start >> 20); k <= 73 + (end >> 20); ++k) {
            bitSet.set(k);
        }
        for (k = 585 + (start >> 17); k <= 585 + (end >> 17); ++k) {
            bitSet.set(k);
        }
        for (k = 4681 + (start >> 14); k <= 4681 + (end >> 14); ++k) {
            bitSet.set(k);
        }
        return bitSet;
    }

    protected List<Chunk> optimizeChunkList(List<Chunk> chunks, long minimumOffset) {
        Chunk lastChunk = null;
        Collections.sort(chunks);
        ArrayList<Chunk> result = new ArrayList<Chunk>();
        for (Chunk chunk : chunks) {
            if (chunk.getChunkEnd() <= minimumOffset) continue;
            if (result.isEmpty()) {
                result.add(chunk);
                lastChunk = chunk;
                continue;
            }
            if (!lastChunk.overlaps(chunk) && !lastChunk.isAdjacentTo(chunk)) {
                result.add(chunk);
                lastChunk = chunk;
                continue;
            }
            if (chunk.getChunkEnd() <= lastChunk.getChunkEnd()) continue;
            lastChunk.setChunkEnd(chunk.getChunkEnd());
        }
        return result;
    }

    private void skipToSequence(int sequenceIndex) {
        for (int i = 0; i < sequenceIndex; ++i) {
            int nBins = this.readInteger();
            for (int j = 0; j < nBins; ++j) {
                int bin = this.readInteger();
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            this.skipBytes(8 * nLinearBins);
        }
    }

    private void readBytes(byte[] bytes) {
        this.mFileBuffer.get(bytes);
    }

    private int readInteger() {
        return this.mFileBuffer.getInt();
    }

    private long readLong() {
        return this.mFileBuffer.getLong();
    }

    private void skipBytes(int count) {
        this.mFileBuffer.position(this.mFileBuffer.position() + count);
    }

    private void seek(int position) {
        this.mFileBuffer.position(position);
    }
}

