/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.cga.tools.sam;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.cmdline.Usage;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMFileWriter;
import net.sf.samtools.SAMFileWriterFactory;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.util.RuntimeEOFException;
import org.broadinstitute.cga.tools.sam.ParallelSAMIterator;

public class PairMaker
extends CommandLineProgram {
    @Usage(programVersion="1.0")
    public String USAGE = "Reconstitutes mate pairs from alignments for individual fragment ends. Individual end alignment files are expected to be sorted by read name. Multiple alignments are allowed and in this case the single best pairing will be selected.";
    @Option(shortName="I1", doc="Input file (bam or sam) with alignments for end 1 (first mate in a pair).", optional=false)
    public File IN1 = null;
    @Option(shortName="I2", doc="Input file (bam or sam) with alignments for end 2 (second mate in a pair).", optional=false)
    public File IN2 = null;
    @Option(shortName="O", optional=false, doc="Output file to write found/selected pairs into.")
    public File OUTPUT = null;
    @Option(shortName="F", doc="Turns on a 'filter' mode: only records/pairs passing the filter will be written into the output file. Filter condition is a logical combination (using parentheses for grouping, & for AND, | for OR, = for specifying values) of the primitives listed below. Primitives that end with  1 or 2 apply specifically to pair end 1 or 2, respectively; when, in addition to primitives <P>1 and <P>2,  a primitive <P> is also defined, it is always interpreted as <P>1 & <P>2. Primitives: PAIRSONLY (print only pairs=both ends mapped), MINQ1=<N>, MINQ2=<N>, MINQ=<N> (minimum alignment quality if read is mapped)  for MINQ1=<N> & MINQ2=<N>), ERRDIFF1=<N>, ERRDIFF2=<N> (when next-best alignment(s) are available", optional=true)
    public String FILTER = null;
    @Option(shortName="Q", optional=true, doc="Minimum mapping quality required on both ends in order to accept the pair.")
    public Integer MINQ = -1;
    @Option(shortName="P", optional=true, doc="Specifies behavior for unpaired reads: KEEP_UNPAIRED, DISCARD_UNPAIRED or DIE_ON_UNPAIRED.")
    public String PAIRED_ACTION = "KEEP_UNPAIRED";
    PairOption paired = null;
    public static int INFINITY = 1000000000;
    private int fragments_seen = 0;
    private int end1_missing = 0;
    private int end2_missing = 0;
    private int end1_unmapped = 0;
    private int end2_unmapped = 0;
    private int end1_unpaired_unmapped = 0;
    private int end2_unpaired_unmapped = 0;
    private int both_unmapped = 0;
    private int both_mapped = 0;
    private int both_unique = 0;
    private int proper_pair = 0;
    private int proper_unique_pair = 0;
    private int outer_pair = 0;
    private int side_pair = 0;
    private int inter_chrom = 0;

    public static void main(String[] argv) {
        System.exit(new PairMaker().instanceMain(argv));
    }

    @Override
    protected int doWork() {
        try {
            this.paired = PairOption.valueOf(this.PAIRED_ACTION);
        }
        catch (Exception e) {
            System.err.println("Unrecognized PAIRED_ACTION value. Allowed options are:");
            for (PairOption p : PairOption.values()) {
                System.out.print(p.toString() + " ");
            }
            System.exit(1);
        }
        SAMFileReader end1Reader = new SAMFileReader(this.IN1);
        SAMFileReader end2Reader = new SAMFileReader(this.IN2);
        SAMFileHeader h = this.checkHeaders(end1Reader.getFileHeader(), end2Reader.getFileHeader());
        SAMFileWriter outWriter = new SAMFileWriterFactory().makeSAMOrBAMWriter(h, true, this.OUTPUT);
        ParallelSAMIterator pi = new ParallelSAMIterator(end1Reader, end2Reader);
        SAMRecord r1 = null;
        SAMRecord r2 = null;
        block18: while (pi.hasNext()) {
            ++this.fragments_seen;
            Object ends = pi.next();
            List end1 = (List)ends.get(0);
            List end2 = (List)ends.get(1);
            if (end1.size() == 0) {
                ++this.end1_missing;
                switch (this.paired) {
                    case DISCARD_UNPAIRED: {
                        continue block18;
                    }
                    case DIE_ON_UNPAIRED: {
                        throw new RuntimeException("End2 read " + ((SAMRecord)end2.get(0)).getReadName() + " is unpaired.");
                    }
                    case KEEP_UNPAIRED: {
                        break;
                    }
                    default: {
                        throw new RuntimeEOFException("Internal error (BUG): unprocessed paired action type: " + (Object)((Object)this.paired));
                    }
                }
                r1 = null;
                r2 = this.selectBestSingleEnd(end2);
                if (PairMaker.isReadUnmapped(r2)) {
                    ++this.end2_unpaired_unmapped;
                }
                this.setPairingInformation(r1, r2);
                outWriter.addAlignment(r2);
                continue;
            }
            if (end2.size() == 0) {
                ++this.end2_missing;
                switch (this.paired) {
                    case DISCARD_UNPAIRED: {
                        continue block18;
                    }
                    case DIE_ON_UNPAIRED: {
                        throw new RuntimeException("End1 read " + ((SAMRecord)end1.get(0)).getReadName() + " is unpaired.");
                    }
                    case KEEP_UNPAIRED: {
                        break;
                    }
                    default: {
                        throw new RuntimeEOFException("Internal error (BUG): unprocessed paired action type: " + (Object)((Object)this.paired));
                    }
                }
                r1 = this.selectBestSingleEnd(end1);
                r2 = null;
                if (PairMaker.isReadUnmapped(r1)) {
                    ++this.end1_unpaired_unmapped;
                }
                this.setPairingInformation(r1, r2);
                outWriter.addAlignment(r1);
                continue;
            }
            if (end1.size() == 1 && end2.size() == 1) {
                r1 = (SAMRecord)end1.get(0);
                r2 = (SAMRecord)end2.get(0);
                if (PairMaker.isReadUnmapped(r1)) {
                    ++this.end1_unmapped;
                    if (PairMaker.isReadUnmapped(r2)) {
                        ++this.end2_unmapped;
                        ++this.both_unmapped;
                    }
                } else if (PairMaker.isReadUnmapped(r2)) {
                    ++this.end2_unmapped;
                } else {
                    ++this.both_mapped;
                    boolean unique = false;
                    if (r1.getMappingQuality() > 0 && r2.getMappingQuality() > 0) {
                        unique = true;
                        ++this.both_unique;
                    }
                    if (r1.getReferenceIndex() != r2.getReferenceIndex()) {
                        ++this.inter_chrom;
                    } else {
                        switch (this.orientation(r1, r2)) {
                            case INNER: {
                                ++this.proper_pair;
                                if (!unique) break;
                                ++this.proper_unique_pair;
                                break;
                            }
                            case OUTER: {
                                ++this.outer_pair;
                                break;
                            }
                            case LEFT: 
                            case RIGHT: {
                                ++this.side_pair;
                                break;
                            }
                        }
                    }
                }
                this.setPairingInformation(r1, r2);
                outWriter.addAlignment(r1);
                outWriter.addAlignment(r2);
                continue;
            }
            if (end1.size() == 1 && PairMaker.isReadUnmapped((SAMRecord)end1.get(0))) {
                r1 = (SAMRecord)end1.get(0);
                r2 = this.selectBestSingleEnd(end2);
                ++this.end1_unmapped;
                if (PairMaker.isReadUnmapped(r2)) {
                    ++this.end2_unmapped;
                    ++this.both_unmapped;
                }
                this.setPairingInformation(r1, r2);
                outWriter.addAlignment(r1);
                outWriter.addAlignment(r2);
                continue;
            }
            if (end2.size() == 1 && PairMaker.isReadUnmapped((SAMRecord)end2.get(0))) {
                r1 = this.selectBestSingleEnd(end1);
                r2 = (SAMRecord)end2.get(0);
                ++this.end2_unmapped;
                if (PairMaker.isReadUnmapped(r1)) {
                    ++this.end1_unmapped;
                    ++this.both_unmapped;
                }
                this.setPairingInformation(r1, r2);
                outWriter.addAlignment(r1);
                outWriter.addAlignment(r2);
                continue;
            }
            SAMRecord[] bestPair = this.selectBestPair(end1, end2);
            r1 = bestPair[0];
            r2 = bestPair[1];
            ++this.both_mapped;
            if (r1.getMappingQuality() > 0 && r2.getMappingQuality() > 0) {
                ++this.both_unique;
            }
            if (r1.getReferenceIndex() == r2.getReferenceIndex() && this.orientation(r1, r2) == PairOrientation.INNER) {
                ++this.proper_pair;
            }
            if (r1.getMappingQuality() > 0 && r2.getMappingQuality() > 0 && r1.getReferenceIndex() == r2.getReferenceIndex() && this.orientation(r1, r2) == PairOrientation.INNER) {
                ++this.proper_unique_pair;
            }
            this.setPairingInformation(r1, r2);
            outWriter.addAlignment(r1);
            outWriter.addAlignment(r2);
        }
        pi.close();
        outWriter.close();
        System.out.println();
        System.out.println("Total fragments (read pairs): " + this.fragments_seen);
        System.out.println("Unpaired end1 reads (end2 missing): " + this.end2_missing);
        System.out.println("Unpaired end1 reads (end2 missing), unmapped: " + this.end1_unpaired_unmapped);
        System.out.println("Unpaired end2 reads (end1 missing): " + this.end1_missing);
        System.out.println("Unpaired end2 reads (end1 missing), unmapped: " + this.end2_unpaired_unmapped);
        System.out.println("Pairs with end1 unmapped (regardless of end2 status): " + this.end1_unmapped);
        System.out.println("Pairs with end2 unmapped (regardless of end1 status): " + this.end2_unmapped);
        System.out.println("Pairs with both ends unmapped: " + this.both_unmapped);
        System.out.println("Pairs with both ends mapped: " + this.both_mapped);
        System.out.println("Pairs with both ends mapped uniquely (MQ>0): " + this.both_unique);
        System.out.println("Pairs with both ends mapped properly: " + this.proper_pair);
        System.out.println("Pairs with both ends mapped uniquely and properly: " + this.proper_unique_pair);
        System.out.println();
        return 0;
    }

    private Query<SAMRecord[]> parseConditions(String filter) {
        filter = filter.trim();
        int level = 0;
        block6: for (int i = 0; i < filter.length(); ++i) {
            switch (filter.charAt(i)) {
                case '(': {
                    ++level;
                    continue block6;
                }
                case ')': {
                    if (--level >= 0) continue block6;
                    throw new RuntimeException("Too many closing parentheses in the expression.");
                }
                case '&': {
                    if (level > 0) continue block6;
                    return new CompositeQuery<SAMRecord[]>(this.parseConditions(filter.substring(0, i)), this.parseConditions(filter.substring(i + 1)), Query.Operator.AND);
                }
                case '|': {
                    if (level > 0) continue block6;
                    return new CompositeQuery<SAMRecord[]>(this.parseConditions(filter.substring(0, i)), this.parseConditions(filter.substring(i + 1)), Query.Operator.OR);
                }
            }
        }
        if (level > 0) {
            throw new RuntimeException("Too many opening parentheses in the expression.");
        }
        if (filter.charAt(0) == '(' && filter.charAt(filter.length() - 1) == ')') {
            return this.parseConditions(filter.substring(1, filter.length() - 1));
        }
        int equal_pos = filter.indexOf(61);
        if (equal_pos < 0 && "PAIRSONLY".equals(filter)) {
            return new BothEndsMappedQuery();
        }
        return null;
    }

    private SAMFileHeader checkHeaders(SAMFileHeader h1, SAMFileHeader h2) {
        if (h1 == null) {
            return h2;
        }
        if (h2 == null) {
            return h1;
        }
        if (!h1.equals(h2)) {
            throw new RuntimeException("Headers in the two input files do not match");
        }
        return h1;
    }

    private SAMRecord selectBestSingleEnd(List<SAMRecord> l) {
        if (l.size() == 0) {
            return null;
        }
        if (l.size() == 1) {
            return l.get(0);
        }
        int best_qual = -1;
        int n_unmapped = 0;
        ArrayList<SAMRecord> best = new ArrayList<SAMRecord>();
        for (SAMRecord r : l) {
            if (r.getNotPrimaryAlignmentFlag()) continue;
            if (r.getReadUnmappedFlag()) {
                ++n_unmapped;
                continue;
            }
            if (r.getMappingQuality() > best_qual) {
                best_qual = r.getMappingQuality();
                best.clear();
                best.add(r);
                continue;
            }
            if (r.getMappingQuality() != best_qual) continue;
            best.add(r);
        }
        if (n_unmapped > 1) {
            throw new RuntimeException("Currently Unsupported: SAM file might be not fully compliant. Multiple 'unmapped' records found for read " + l.get(0).getReadName());
        }
        if (best.size() == 0) {
            throw new RuntimeException("Currently Unsupported: SAM file might be not fully compliant. No primary alignments found for read " + l.get(0).getReadName());
        }
        if (best.size() == 1) {
            return (SAMRecord)best.get(0);
        }
        if (best_qual != 0) {
            throw new RuntimeException("Multiple alignments for the same read found with non-zero score. Read: " + l.get(0).getReadName() + " best score: " + best_qual);
        }
        return (SAMRecord)best.get((int)(Math.random() * (double)best.size()));
    }

    private SAMRecord[] selectBestPair(List<SAMRecord> end1, List<SAMRecord> end2) {
        SAMRecord r1 = this.selectBestSingleEnd(end1);
        SAMRecord r2 = this.selectBestSingleEnd(end2);
        if (PairMaker.isReadUnmapped(r1) || PairMaker.isReadUnmapped(r2)) {
            throw new RuntimeException("Unmapped read in selectBestPair: should never happen. read1=" + r1.toString() + "; read2=" + r2.toString());
        }
        if (r1.getMappingQuality() > 0 && r2.getMappingQuality() > 0) {
            return new SAMRecord[]{r1, r2};
        }
        ArrayList<SAMRecord> toChooseFrom = new ArrayList<SAMRecord>();
        ArrayList<SAMRecord> toChooseFromAll = new ArrayList<SAMRecord>();
        if (r1.getMappingQuality() > 0) {
            for (SAMRecord r : end2) {
                if (r.getNotPrimaryAlignmentFlag()) continue;
                if (r.getReferenceIndex().intValue() == r1.getReferenceIndex().intValue()) {
                    toChooseFrom.add(r);
                    continue;
                }
                toChooseFromAll.add(r);
            }
            if (toChooseFrom.size() == 1) {
                return new SAMRecord[]{r1, (SAMRecord)toChooseFrom.get(0)};
            }
            if (toChooseFrom.size() > 1) {
                return new SAMRecord[]{r1, (SAMRecord)toChooseFrom.get((int)(Math.random() * (double)toChooseFrom.size()))};
            }
            if (toChooseFromAll.size() == 0) {
                throw new RuntimeException("No primary alignments found for read " + end2.get(0).getReadName());
            }
            return new SAMRecord[]{r1, (SAMRecord)toChooseFromAll.get((int)(Math.random() * (double)toChooseFromAll.size()))};
        }
        if (r2.getMappingQuality() > 0) {
            for (SAMRecord r : end1) {
                if (r.getNotPrimaryAlignmentFlag()) continue;
                if (r.getReferenceIndex().intValue() == r2.getReferenceIndex().intValue()) {
                    toChooseFrom.add(r);
                    continue;
                }
                toChooseFromAll.add(r);
            }
            if (toChooseFrom.size() == 1) {
                return new SAMRecord[]{(SAMRecord)toChooseFrom.get(0), r2};
            }
            if (toChooseFrom.size() > 1) {
                return new SAMRecord[]{(SAMRecord)toChooseFrom.get((int)(Math.random() * (double)toChooseFrom.size())), r2};
            }
            if (toChooseFromAll.size() == 0) {
                throw new RuntimeException("No primary alignments found for read " + end1.get(0).getReadName());
            }
            return new SAMRecord[]{(SAMRecord)toChooseFromAll.get((int)(Math.random() * (double)toChooseFromAll.size())), r2};
        }
        ArrayList<SAMRecord[]> toChooseFromP = new ArrayList<SAMRecord[]>();
        ArrayList<SAMRecord[]> toChooseFromPAll = new ArrayList<SAMRecord[]>();
        for (SAMRecord rr1 : end1) {
            if (rr1.getNotPrimaryAlignmentFlag()) continue;
            for (SAMRecord rr2 : end2) {
                if (rr2.getNotPrimaryAlignmentFlag()) continue;
                SAMRecord[] z = new SAMRecord[]{rr1, rr2};
                if (rr1.getReferenceIndex().intValue() == rr2.getReferenceIndex().intValue()) {
                    toChooseFromP.add(z);
                }
                toChooseFromPAll.add(z);
            }
        }
        if (toChooseFromP.size() == 1) {
            return (SAMRecord[])toChooseFromP.get(0);
        }
        if (toChooseFromP.size() > 1) {
            return (SAMRecord[])toChooseFromP.get((int)(Math.random() * (double)toChooseFromP.size()));
        }
        if (toChooseFromPAll.size() == 0) {
            throw new RuntimeException("No primary alignments found for read " + end1.get(0).getReadName());
        }
        return (SAMRecord[])toChooseFromPAll.get((int)(Math.random() * (double)toChooseFromPAll.size()));
    }

    private SAMRecord selectUniqueSingleEnd(List<SAMRecord> l, int minq) {
        if (l.size() == 0) {
            return null;
        }
        if (l.size() == 1) {
            if (l.get(0).getMappingQuality() >= minq) {
                return l.get(0);
            }
            return null;
        }
        int n_unmapped = 0;
        ArrayList<SAMRecord> best = new ArrayList<SAMRecord>();
        for (SAMRecord r : l) {
            if (PairMaker.isReadUnmapped(r)) {
                ++n_unmapped;
                continue;
            }
            if (r.getMappingQuality() < minq) continue;
            best.add(r);
        }
        if (best.size() == 0) {
            return null;
        }
        if (best.size() > 1) {
            for (SAMRecord r : best) {
                System.out.println("READ " + r.getReadName() + " mapQ=" + r.getMappingQuality() + " at=" + r.getReferenceName() + ":" + r.getAlignmentStart() + "(" + (r.getReadNegativeStrandFlag() ? "-" : "+") + ") cig=" + r.getCigarString());
            }
            throw new RuntimeException("Multiple alignments for read " + l.get(0).getReadName() + ", all with Q>=" + minq);
        }
        return (SAMRecord)best.get(0);
    }

    private void setPairingInformation(SAMRecord r1, SAMRecord r2) {
        if (r1 == null && r2 == null) {
            throw new RuntimeException("Both ends of the mate pair are passed as 'null'");
        }
        if (r2 == null) {
            r1.setReadPairedFlag(false);
            r1.setMateReferenceIndex(-1);
            r1.setMateNegativeStrandFlag(false);
            r1.setMateAlignmentStart(0);
            return;
        }
        if (r1 == null) {
            r2.setReadPairedFlag(false);
            r2.setMateReferenceIndex(-1);
            r2.setMateNegativeStrandFlag(false);
            r2.setMateAlignmentStart(0);
            return;
        }
        r1.setReadPairedFlag(true);
        r2.setReadPairedFlag(true);
        boolean r1unmapped = PairMaker.isReadUnmapped(r1);
        boolean r2unmapped = PairMaker.isReadUnmapped(r2);
        r1.setMateUnmappedFlag(r2unmapped);
        r1.setMateReferenceIndex(r2unmapped ? -1 : r2.getReferenceIndex());
        r1.setMateNegativeStrandFlag(r2unmapped ? false : r2.getReadNegativeStrandFlag());
        r1.setMateAlignmentStart(r2unmapped ? 0 : r2.getAlignmentStart());
        r1.setFirstOfPairFlag(true);
        r1.setSecondOfPairFlag(false);
        r2.setMateUnmappedFlag(r1unmapped);
        r2.setMateReferenceIndex(r1unmapped ? -1 : r1.getReferenceIndex());
        r2.setMateNegativeStrandFlag(r1unmapped ? false : r1.getReadNegativeStrandFlag());
        r2.setMateAlignmentStart(r1unmapped ? 0 : r1.getAlignmentStart());
        r2.setFirstOfPairFlag(false);
        r2.setSecondOfPairFlag(true);
    }

    private int fragmentSize(SAMRecord r1, SAMRecord r2) {
        if (r1 == null || PairMaker.isReadUnmapped(r1) || r2 == null || PairMaker.isReadUnmapped(r2) || r1.getReferenceIndex() != r2.getReferenceIndex()) {
            return INFINITY;
        }
        if (r1.getAlignmentStart() <= r2.getAlignmentStart()) {
            return r2.getAlignmentStart() + r2.getReadLength() - r1.getAlignmentStart();
        }
        return r1.getAlignmentStart() + r1.getReadLength() - r2.getAlignmentStart();
    }

    private PairOrientation orientation(SAMRecord r1, SAMRecord r2) {
        SAMRecord right;
        SAMRecord left;
        if (r1 == null || r2 == null || PairMaker.isReadUnmapped(r1) || PairMaker.isReadUnmapped(r2)) {
            return PairOrientation.NONE;
        }
        if (r1.getReferenceIndex() == r2.getReferenceIndex()) {
            if (r1.getAlignmentStart() <= r2.getAlignmentStart()) {
                left = r1;
                right = r2;
            } else {
                left = r2;
                right = r1;
            }
        } else if (r1.getReferenceIndex() < r2.getReferenceIndex()) {
            left = r1;
            right = r2;
        } else {
            left = r2;
            right = r1;
        }
        if (!left.getReadNegativeStrandFlag()) {
            if (right.getReadNegativeStrandFlag()) {
                return PairOrientation.INNER;
            }
            return PairOrientation.RIGHT;
        }
        if (right.getReadNegativeStrandFlag()) {
            return PairOrientation.LEFT;
        }
        return PairOrientation.OUTER;
    }

    private double pairingScore(SAMRecord r1, SAMRecord r2) {
        return (double)(r1.getMappingQuality() + r2.getMappingQuality()) * Math.exp(1.0);
    }

    public static boolean isReadUnmapped(SAMRecord r) {
        if (r.getReadUnmappedFlag()) {
            return true;
        }
        return (r.getReferenceIndex() == null || r.getReferenceIndex() == -1) && (r.getReferenceName() == null || r.getReferenceName().equals("*")) || r.getAlignmentStart() == 0;
    }

    class BothEndsMappedQuery
    implements Query<SAMRecord[]> {
        BothEndsMappedQuery() {
        }

        @Override
        public boolean isSatisfied(SAMRecord[] p) {
            return !PairMaker.isReadUnmapped(p[0]) && !PairMaker.isReadUnmapped(p[1]);
        }
    }

    class CompositeQuery<T>
    implements Query<T> {
        private Query<T> q1;
        private Query<T> q2;
        private Query.Operator type;

        CompositeQuery(Query<T> q1, Query<T> q2, Query.Operator type) {
            this.q1 = q1;
            this.q2 = q2;
            this.type = type;
        }

        @Override
        public boolean isSatisfied(T record) {
            switch (this.type) {
                case AND: {
                    return this.q1.isSatisfied(record) && this.q2.isSatisfied(record);
                }
                case OR: {
                    return this.q1.isSatisfied(record) || this.q2.isSatisfied(record);
                }
            }
            throw new IllegalStateException("Unknown composite query operator");
        }
    }

    static interface Query<T> {
        public boolean isSatisfied(T var1);

        public static enum Operator {
            OR,
            AND;

        }
    }

    class Pairing {
        SAMRecord r1;
        SAMRecord r2;
        double score;

        Pairing() {
            this(null, null, INFINITY);
        }

        Pairing(SAMRecord r1, SAMRecord r2) {
            this(r1, r2, INFINITY);
        }

        Pairing(SAMRecord r1, SAMRecord r2, double score) {
            this.r1 = r1;
            this.r2 = r2;
            this.score = score;
        }

        SAMRecord getFirst() {
            return this.r1;
        }

        SAMRecord getSecond() {
            return this.r2;
        }

        double getScore() {
            return this.score;
        }

        void setFirst(SAMRecord r) {
            this.r1 = r;
        }

        void setSecond(SAMRecord r) {
            this.r2 = r;
        }

        void setScore(double score) {
            this.score = score;
        }
    }

    static enum PairOrientation {
        INNER,
        OUTER,
        LEFT,
        RIGHT,
        NONE;

    }

    static enum PairOption {
        KEEP_UNPAIRED,
        DISCARD_UNPAIRED,
        DIE_ON_UNPAIRED;

    }
}

