/*
 * Decompiled with CFR 0.152.
 */
package edu.unc.genomics.converters;

import com.beust.jcommander.Parameter;
import edu.unc.genomics.CommandLineToolException;
import edu.unc.genomics.Interval;
import edu.unc.genomics.ReadablePathValidator;
import edu.unc.genomics.WigMathTool;
import edu.unc.genomics.io.WigFileException;
import edu.unc.genomics.io.WigFileReader;
import java.io.IOException;
import java.nio.file.Path;
import org.apache.log4j.Logger;

public class InterpolateDiscontinuousData
extends WigMathTool {
    private static final Logger log = Logger.getLogger(InterpolateDiscontinuousData.class);
    @Parameter(names={"-i", "--input"}, description="Input file (Wig)", required=true, validateWith=ReadablePathValidator.class)
    public Path inputFile;
    @Parameter(names={"-t", "--type"}, description="Interpolant (nearest/linear/cubic)")
    public String type = "linear";
    @Parameter(names={"-m", "--max"}, description="Maximum span of missing values to interpolate (bp)")
    public int max = 1000;
    private WigFileReader wig;
    private Interpolant interp;

    @Override
    public void setup() {
        this.interp = Interpolant.fromName(this.type);
        if (this.interp == null) {
            log.error((Object)("Unknown interpolation method: " + this.type));
            throw new CommandLineToolException("Unknown interpolation method: " + this.type + ". Options are nearest, linear, cubic");
        }
        log.debug((Object)("Using interpolant: " + this.type));
        if (this.max > 10000000) {
            log.warn((Object)"Allowable span of missing values exceeds processing chunk size");
        }
        try {
            this.wig = WigFileReader.autodetect((Path)this.inputFile);
        }
        catch (IOException e) {
            log.error((Object)"Error initializing (Big)Wig file");
            e.printStackTrace();
            throw new CommandLineToolException("Error initializing (Big)Wig file");
        }
        this.inputs.add(this.wig);
    }

    @Override
    public float[] compute(Interval chunk) throws IOException, WigFileException {
        float y1;
        float y0;
        int x1;
        int x0;
        int j;
        int queryStop;
        float[] result = this.wig.query(chunk).getValues();
        int nansbefore = 0;
        for (float v : result) {
            if (!Float.isNaN(v)) continue;
            ++nansbefore;
        }
        log.debug((Object)("Chunk has " + nansbefore + " missing values before interpolation"));
        int first = 0;
        while (Float.isNaN(result[first]) && ++first < result.length) {
        }
        int last = result.length - 1;
        while (Float.isNaN(result[last]) && --last > first) {
        }
        if (first == result.length) {
            log.warn((Object)("Skipping entire chunk " + chunk + " that is missing values"));
            return result;
        }
        if (first > 0 && first < this.max) {
            int upstream = this.max - first;
            int queryStart = Math.max(chunk.getStart() - upstream, this.wig.getChrStart(chunk.getChr()));
            if (queryStart >= (queryStop = chunk.getStart() - 1)) {
                log.debug((Object)("Cannot interpolate missing values at the start of chromosome " + chunk.getChr()));
            } else {
                float[] upstreamResult = this.wig.query(chunk.getChr(), queryStart, queryStop).getValues();
                j = upstreamResult.length - 1;
                while (Float.isNaN(upstreamResult[j]) && --j >= 0) {
                }
                if (j >= 0 && !Float.isNaN(upstreamResult[j])) {
                    x0 = -j;
                    x1 = first;
                    y0 = upstreamResult[j];
                    y1 = result[first];
                    this.doInterpolation(result, x0, x1, y0, y1);
                }
            }
        }
        if (last < result.length - 1 && last >= result.length - this.max) {
            int downstream = this.max - (result.length - last);
            int queryStart = chunk.getStop() + 1;
            if (queryStart >= (queryStop = Math.min(chunk.getStop() + downstream, this.wig.getChrStop(chunk.getChr())))) {
                log.debug((Object)("Cannot interpolate missing values at the end of chromosome " + chunk.getChr()));
            } else {
                float[] downstreamResult = this.wig.query(chunk.getChr(), queryStart, queryStop).getValues();
                j = 0;
                while (Float.isNaN(downstreamResult[j]) && ++j < downstreamResult.length) {
                }
                if (j < downstreamResult.length && !Float.isNaN(downstreamResult[j])) {
                    x0 = last;
                    x1 = result.length + j;
                    y0 = result[last];
                    y1 = downstreamResult[j];
                    this.doInterpolation(result, x0, x1, y0, y1);
                }
            }
        }
        for (int i = first; i < last; ++i) {
            if (!Float.isNaN(result[i])) continue;
            int x02 = i - 1;
            while (Float.isNaN(result[++i])) {
            }
            int x12 = i;
            if (x12 - x02 <= this.max) {
                this.doInterpolation(result, x02, x12, result[x02], result[x12]);
                continue;
            }
            log.debug((Object)("Skipping interval " + chunk.getChr() + ":" + (chunk.getStart() + x02) + "-" + (chunk.getStart() + x12) + " (exceeds maximum span)"));
        }
        int nansafter = 0;
        for (float v : result) {
            if (!Float.isNaN(v)) continue;
            ++nansafter;
        }
        log.debug((Object)("Chunk has " + nansafter + " missing values after interpolation"));
        return result;
    }

    protected void doInterpolation(float[] x, int x0, int x1, float y0, float y1) {
        switch (this.interp) {
            case NEAREST: {
                int i;
                int center = (x0 + x1) / 2;
                for (i = x0 + 1; i <= center; ++i) {
                    if (i <= 0 || i >= x.length) continue;
                    x[i] = y0;
                }
                i = center + 1;
                while ((float)i < y0) {
                    if (i > 0 && i < x.length) {
                        x[i] = y1;
                    }
                    ++i;
                }
                break;
            }
            case LINEAR: {
                float m = (y1 - y0) / (float)(x1 - x0);
                for (int i = x0 + 1; i < x1; ++i) {
                    if (i <= 0 || i >= x.length) continue;
                    x[i] = y0 + m * (float)(i - x0);
                }
                break;
            }
            case CUBIC: {
                int l = x1 - x0;
                float a = (y1 - y0) / (float)(l * l);
                float b = 2.0f * (y0 - y1) / (float)(l * l * l);
                for (int i = x0 + 1; i < x1; ++i) {
                    if (i <= 0 || i >= x.length) continue;
                    x[i] = y0 + a * (float)(i - x0) * (float)(i - x0) + b * (float)(i - x0) * (float)(i - x0) * (float)(i - x1);
                }
                break;
            }
        }
    }

    public static void main(String[] args) throws IOException, WigFileException {
        new InterpolateDiscontinuousData().instanceMain(args);
    }

    public static enum Interpolant {
        NEAREST("nearest"),
        LINEAR("linear"),
        CUBIC("cubic");

        private String name;

        private Interpolant(String name) {
            this.name = name;
        }

        public static Interpolant fromName(String name) {
            for (Interpolant i : Interpolant.values()) {
                if (!i.getName().equalsIgnoreCase(name)) continue;
                return i;
            }
            return null;
        }

        public String getName() {
            return this.name;
        }
    }
}

