/*
 * Decompiled with CFR 0.152.
 */
package edu.unc.genomics.wigmath;

import com.beust.jcommander.Parameter;
import edu.unc.genomics.CommandLineTool;
import edu.unc.genomics.CommandLineToolException;
import edu.unc.genomics.io.WigFile;
import edu.unc.genomics.io.WigFileException;
import edu.unc.genomics.wigmath.WigMathTool;
import edu.unc.utils.FloatCorrelation;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import org.apache.log4j.Logger;
import org.broad.igv.bbfile.WigItem;

public class WigCorrelate
extends CommandLineTool {
    private static final Logger log = Logger.getLogger(WigCorrelate.class);
    @Parameter(description="Input files", required=true)
    public List<String> inputFiles = new ArrayList<String>();
    @Parameter(names={"-w", "--window"}, description="Window size (bp)")
    public int windowSize = 100;
    @Parameter(names={"-s", "--step"}, description="Sliding shift step (bp)")
    public int stepSize = 50;
    @Parameter(names={"-t", "--type"}, description="Correlation metric to use (pearson/spearman)")
    public String type = "pearson";
    @Parameter(names={"-o", "--output"}, description="Output file")
    public Path outputFile;
    private Correlation corr;
    private List<WigFile> wigs = new ArrayList<WigFile>();
    private List<String> chromosomes;
    int[] chrStarts;
    int[] chrStops;
    int[] chrLengths;
    int[] nBins;
    int totalNumBins = 0;
    private float[][] correlationMatrix;

    @Override
    public void run() throws IOException {
        int i;
        int i2;
        if (this.inputFiles.size() < 2) {
            throw new CommandLineToolException("Cannot correlate < 2 input files.");
        }
        this.corr = Correlation.fromName(this.type);
        if (this.corr == null) {
            log.error((Object)("Unknown correlation metric: " + this.type));
            throw new CommandLineToolException("Unknown correlation metric: " + this.type + ". Options are pearson, spearman");
        }
        log.debug((Object)("Computing " + this.type + " correlation"));
        log.debug((Object)"Initializing input files");
        for (String inputFile : this.inputFiles) {
            try {
                this.wigs.add(WigFile.autodetect((Path)Paths.get(inputFile, new String[0])));
            }
            catch (WigFileException | IOException e) {
                log.error((Object)("Error initializing input Wig file: " + inputFile));
                e.printStackTrace();
                throw new CommandLineToolException(e.getMessage());
            }
        }
        log.debug((Object)("Initialized " + this.wigs.size() + " input files"));
        this.correlationMatrix = new float[this.wigs.size()][this.wigs.size()];
        for (i2 = 0; i2 < this.correlationMatrix.length; ++i2) {
            this.correlationMatrix[i2][i2] = 1.0f;
        }
        this.chromosomes = new ArrayList<String>(WigMathTool.getCommonChromosomes(this.wigs));
        this.chrStarts = new int[this.chromosomes.size()];
        Arrays.fill(this.chrStarts, Integer.MAX_VALUE);
        this.chrStops = new int[this.chromosomes.size()];
        Arrays.fill(this.chrStops, Integer.MIN_VALUE);
        for (i2 = 0; i2 < this.chromosomes.size(); ++i2) {
            String chr = this.chromosomes.get(i2);
            for (WigFile w : this.wigs) {
                if (w.getChrStart(chr) < this.chrStarts[i2]) {
                    this.chrStarts[i2] = w.getChrStart(chr);
                }
                if (w.getChrStop(chr) <= this.chrStops[i2]) continue;
                this.chrStops[i2] = w.getChrStop(chr);
            }
        }
        this.chrLengths = new int[this.chromosomes.size()];
        this.nBins = new int[this.chromosomes.size()];
        for (i2 = 0; i2 < this.chromosomes.size(); ++i2) {
            this.chrLengths[i2] = this.chrStops[i2] - this.chrStarts[i2] + 1;
            this.nBins[i2] = this.chrLengths[i2] / this.stepSize;
            if (this.nBins[i2] * this.stepSize != this.chrLengths[i2]) {
                int n = i2;
                this.nBins[n] = this.nBins[n] + 1;
            }
            this.totalNumBins += this.nBins[i2];
        }
        log.debug((Object)("Total number of bins for all chromosomes = " + this.totalNumBins));
        for (i2 = 0; i2 < this.wigs.size() - 1; ++i2) {
            log.debug((Object)("Loading data from file " + i2));
            float[] binsI = this.getDataVector(this.wigs.get(i2));
            for (int j = i2 + 1; j < this.wigs.size(); ++j) {
                log.debug((Object)("Loading data from file " + j));
                float[] binsJ = this.getDataVector(this.wigs.get(j));
                log.debug((Object)("Correlating (" + i2 + "," + j + ")"));
                switch (this.corr) {
                    case PEARSON: {
                        this.correlationMatrix[i2][j] = FloatCorrelation.pearson(binsI, binsJ);
                        break;
                    }
                    case SPEARMAN: {
                        this.correlationMatrix[i2][j] = FloatCorrelation.spearman(binsI, binsJ);
                    }
                }
                this.correlationMatrix[j][i2] = this.correlationMatrix[i2][j];
            }
        }
        StringBuilder output = new StringBuilder(this.type);
        for (i = 0; i < this.wigs.size(); ++i) {
            output.append("\t" + this.wigs.get(i).getPath().getFileName());
        }
        for (i = 0; i < this.wigs.size(); ++i) {
            output.append("\n" + this.wigs.get(i).getPath().getFileName());
            for (int j = 0; j < this.wigs.size(); ++j) {
                output.append("\t" + this.correlationMatrix[i][j]);
            }
        }
        if (this.outputFile != null) {
            log.debug((Object)"Writing to output file");
            try (BufferedWriter writer = Files.newBufferedWriter(this.outputFile, Charset.defaultCharset(), new OpenOption[0]);){
                writer.write(output.toString());
            }
        } else {
            System.out.println(output.toString());
        }
    }

    private float[] getDataVector(WigFile w) {
        int i;
        float[] values = new float[this.totalNumBins];
        int[] counts = new int[this.totalNumBins];
        int binOffset = 0;
        for (i = 0; i < this.chromosomes.size(); ++i) {
            String chr = this.chromosomes.get(i);
            try {
                Iterator result = w.query(chr, w.getChrStart(chr), w.getChrStop(chr));
                while (result.hasNext()) {
                    WigItem item = (WigItem)result.next();
                    int bin = item.getStartBase() / this.stepSize;
                    for (int binStart = bin * this.stepSize + 1; binStart <= item.getEndBase(); binStart += this.stepSize) {
                        int binEnd = binStart + this.windowSize - 1;
                        int intersectStart = Math.max(binStart, item.getStartBase());
                        int intersectStop = Math.min(binEnd, item.getEndBase());
                        int overlap = intersectStop - intersectStart + 1;
                        int n = bin + binOffset;
                        values[n] = values[n] + (float)overlap * item.getWigValue();
                        int n2 = bin + binOffset;
                        counts[n2] = counts[n2] + overlap;
                        ++bin;
                    }
                }
            }
            catch (WigFileException | IOException e) {
                log.error((Object)("Error getting data from wig file: " + w.getPath()));
                throw new CommandLineToolException(e.getMessage());
            }
            binOffset += this.nBins[i];
        }
        for (i = 0; i < this.totalNumBins; ++i) {
            if (counts[i] > 0) {
                int n = i;
                values[n] = values[n] / (float)counts[i];
                continue;
            }
            values[i] = Float.NaN;
        }
        return values;
    }

    public static void main(String[] args) throws IOException, WigFileException {
        new WigCorrelate().instanceMain(args);
    }

    public static enum Correlation {
        PEARSON("pearson"),
        SPEARMAN("spearman");

        private String name;

        private Correlation(String name) {
            this.name = name;
        }

        public static Correlation fromName(String name) {
            for (Correlation c : Correlation.values()) {
                if (!c.getName().equalsIgnoreCase(name)) continue;
                return c;
            }
            return null;
        }

        public String getName() {
            return this.name;
        }
    }
}

