#!/usr/bin/env python

# Copyright INRA (Institut National de la Recherche Agronomique)
# http://www.inra.fr
# http://urgi.versailles.inra.fr
#
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software.  You can  use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability.
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and,  more generally, to use and operate it in the
# same conditions as regards security.
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.

from commons.core.LoggerFactory             import LoggerFactory
from commons.core.utils.RepetOptionParser   import RepetOptionParser
from commons.core.checker.RepetException    import RepetException
from commons.core.utils.FileUtils           import FileUtils
from commons.core.parsing.GtfParser import GtfParser
import os
LOG_NAME = "TEiso"

class CufflinksGTFToBed(object):
    
    def __init__(self, inputFile = "", outputFile = "", verbosity = 3):
        self._inputFile = inputFile
        self._outputFile = outputFile
        self._verbosity = verbosity
        self._log = LoggerFactory.createLogger("%s.%s" % (LOG_NAME, self.__class__.__name__), self._verbosity)
            
    def setAttributesFromCmdLine(self):
        self._toolVersion = "1.1.a"
        description = "CufflinksGTFToBed version %s" % self._toolVersion
        epilog = "\n parses a GTF file of Cufflinks and create a bed file. \n"
        epilog += "example: CufflinksGTFToBed.py -i <inputFile> -o  <outputFile>\n"
        parser = RepetOptionParser(description = description, epilog = epilog, version = self._toolVersion) 
        parser.add_option("-i", "--inputFile",  dest = "inputFile",  action = "store", type = "string", help = "Input GTF File name (transcript.gtf of Cufflinks).",  default = "")
        parser.add_option("-o", "--outputFile", dest = "outputFile", action = "store", type = "string", help = "output Bed File name", default = "")
        parser.add_option("-v", "--verbosity",  dest = "verbosity",  action = "store", type = "int",    help = "Verbosity [optional] [default: 3]",default = 3)
        options = parser.parse_args()[0]
        self._setAttributesFromOptions(options)

    def _setAttributesFromOptions(self, options):
        self._inputFile  = options.inputFile
        self._outputFile = options.outputFile
        self._verbosity  = options.verbosity
                    
    def _logAndRaise(self, errorMsg):
        self._log.error(errorMsg)
        raise RepetException(errorMsg)

    def checkoption(self):
        if self._outputFile == "":
            #self._log.info("Missing output file destination")
            self._outputFile = "%s.bed" % os.path.splitext(self._inputFile)[0]
        else:
            if FileUtils.isRessourceExists(self._outputFile):
                self._log.info("Output file '%s' already exists!" % self._outputFile)
            
        if self._inputFile == "":
            self._log.info("Missing input file")
        
    def getTranscriptToBed (self, inputFile ,outputFile):
        try:
            filewrite=open(outputFile, "w")
            gtfParser = GtfParser(self._inputFile, assemblyTools=True)
            for transcript in gtfParser.getIterator():
                if(transcript.getDirection()==1):
                    strand="+"
                else:
                    strand="-"
                filewrite.write("%s\t%s\t%s\t%s\t%s\t%s\t%.3f\n"  % (transcript.getChromosome(),transcript.getStart(),
                transcript.getEnd(), transcript.getTagValue("ID"), transcript.getTagValue("gene_id"),
                strand,float(transcript.getTagValue("FPKM")) ))
            filewrite.close()
        except:
                raise Exception("Couldn't open %s for writing" % outputFile)
                
    def run(self):
        self.checkoption()
        self.getTranscriptToBed(self._inputFile, self._outputFile)
        

if __name__== "__main__":
    iTranscriptToBed = CufflinksGTFToBed()
    iTranscriptToBed.setAttributesFromCmdLine()
    iTranscriptToBed.run()


