"""Common functions. """
import logging
import pysam


class BAMProfilesError(Exception):
    pass


class BAMFileError(Exception):
    pass


class TranscriptError(Exception):
    pass


class RNACountsError(Exception):
    """For errors related to RNA Coverage generation using bedtools. """
    pass


def check_bam_file(bam_file):
    """Check if bam file is indexed. If not, index. """
    with pysam.AlignmentFile(bam_file, 'rb') as bam_fileobj:
        try:
            bam_fileobj.fetch(bam_fileobj.references[0])
        except ValueError as err:
            if err.message == 'fetch called on bamfile without index':
                bam_fileobj.close()
                logging.debug('Indexing bam file {}'.format(bam_file))
                pysam.index(bam_file)


def get_ribo_counts(ribo_fileobj, transcript_name, read_length=None, read_offset=0):
    """Get total reads and read counts in all 3 frames for the give transcript
    from input BAM file (indexed).

    """
    read_counts = {}
    total_reads = 0

    if transcript_name not in ribo_fileobj.references:
        raise BAMFileError('Transcript "{}" does not exist in BAM '
                           'file. '.format(transcript_name))

    for record in ribo_fileobj.fetch(transcript_name):
        if read_length:
            if record.query_length != read_length:
                continue
        total_reads += 1
        position = record.pos + 1

        try:
            read_counts[position]
        except KeyError:
            read_counts[position] = {1: 0, 2: 0, 3: 0}

        rem = position % 3
        if rem == 0:
            read_counts[position][3] += 1
        else:
            read_counts[position][rem] += 1

    return read_counts, total_reads

