# check dependencies
missing_deps = []
try:
    import pysam
except ImportError:
    missing_deps.append('pysam')

if len(missing_deps):
    sys.exit('''\nCould not import the following modules. Please check if they
    (or their dependencies are installed correctly.
    \n\n{}\n'''.format('\n'.join(missing_deps)))

import os
import sys
import timeit
import shutil
import zipfile
import logging
import argparse
from datetime import datetime

import core
import config

# Default is production
CONFIG = config.ProductionConfig()


def create_parser():
    """Argument parser. """
    parser = argparse.ArgumentParser(
        prog='ribograph.counts', description='Output read counts for all transcripts')

    # required arguments
    required = parser.add_argument_group('required arguments')
    required.add_argument('--ribo_file', help='Ribo-Seq alignment file in BAM format', required=True)
    required.add_argument('--transcriptome_fasta', help='FASTA format file of the transcriptome', required=True)

    # optional arguments
    parser.add_argument('--read_length', help='Read length to consider (default: %(default)s)',
                        metavar='INTEGER', type=int)
    parser.add_argument('--read_offset', help='Read offset (default: %(default)s)',
                        metavar='INTEGER', type=int, default=0)
    parser.add_argument('--html_file', help='Output file for results (HTML)', default='ribocount.html')
    parser.add_argument('--output_path', help='Files are saved in this directory', default='output')
    parser.add_argument('--debug', help='Flag. Produce debug output', action='store_true')

    return parser


def main():
    parsed = create_parser()
    args = parsed.parse_args()

    if args.debug:
        level = logging.DEBUG
    else:
        level = logging.INFO

    logging.basicConfig(format='%(asctime)s: %(levelname)s %(message)s',
                        level=level, stream=sys.stdout, datefmt='%d/%m/%Y %I:%M:%S %p')

    start = datetime.now()
    logging.debug('Start')
    logging.debug('Supplied Arguments')
    logging.debug('\n{}'.format('\n'.join(['{:<20}: {}'.format(k, v) for k, v in vars(args).items()])))

    (ribo_file, transcriptome_fasta, read_length, read_offset) = (
        args.ribo_file, args.transcriptome_fasta, args.read_length, args.read_offset)

    logging.debug('Checking if BAM file is indexed')
    core.check_bam_file(ribo_file)

    logging.debug('Get transcript information')
    fasta_file = pysam.FastaFile(transcriptome_fasta)

    count = 0
    table_content = ''

    if not os.path.exists(args.output_path):
        os.mkdir(args.output_path)

    output_path = os.path.join(args.output_path, 'ribocount_output')
    if not os.path.exists(output_path):
        os.mkdir(output_path)

    csv_dir = os.path.join(output_path, 'csv')
    if not os.path.exists(csv_dir):
        os.mkdir(csv_dir)

    bam_fileobj = pysam.AlignmentFile(ribo_file, 'rb')
    for transcript in fasta_file.references:
        rp_counts, rp_reads = core.get_ribo_counts(bam_fileobj, transcript, read_length, read_offset)

        if not rp_reads:
            continue

        logging.debug('Writing counts for {}'.format(transcript))
        count += 1
        csv_file = 'RiboCounts{}.csv'.format(count)
        with open(os.path.join(csv_dir, csv_file), 'w') as f:
            f.write('"Position","Frame 1","Frame 2","Frame 3"\n')

            for pos in range(1, len(fasta_file[transcript]) + 1):
                if pos in rp_counts:
                    f.write('{0},{1},{2},{3}\n'.format(
                        pos, rp_counts[pos][1], rp_counts[pos][2], rp_counts[pos][3]))
                else:
                    f.write('{0},{1},{2},{3}\n'.format(pos, 0, 0, 0))
        if count % 2 == 0:
            table_content += '<tr>'
        else:
            table_content += '<tr class="odd">'
        table_content += '<td>{0}</td><td>{1}</td><td>{2}</td><td><a href="csv/{3}">{3}</a></td></tr>'.format(
            count, transcript, rp_reads, csv_file)
    bam_fileobj.close()

    if not read_length:
        read_length = 'All'

    duration = str(datetime.now() - start).split('.')[0]
    logging.debug('Time taken for generating counts {}'.format(duration))

    with open(os.path.join(CONFIG.DATA_DIR, 'ribocount.html')) as g,\
            open(os.path.join(output_path, 'index.html'), 'w') as h:
        h.write(g.read().format(count=count, length=read_length, table_content=table_content, duration=duration))

    for asset in ('css', 'js'):
        asset_dir = os.path.join(output_path, asset)
        if not os.path.exists(asset_dir):
            os.mkdir(asset_dir)
        asset_data_dir = os.path.join(CONFIG.DATA_DIR, asset)
        for fname in os.listdir(asset_data_dir):
            shutil.copy(os.path.join(asset_data_dir, fname),
                        os.path.join(output_path, asset, fname))

    logging.debug('Creating zip file')
    os.chdir(args.output_path)
    with zipfile.ZipFile('ribocount_output.zip', 'w') as zip:
        for root, d, f in os.walk('ribocount_output'):
            for name in f:
                zip.write(os.path.join(root, name))

    logging.debug('Writing HTML report')

    with open(os.path.join(CONFIG.DATA_DIR, 'ribocount_index.html')) as j, open(args.html_file, 'w') as k:
        k.write(j.read().format(count=count, read_length=read_length))

if __name__ == '__main__':
    print timeit.timeit('main()', number=1, setup='from __main__ import main')
