#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""
Cluster the data into regions (defined by size and overlap with next region) and keep only highest peaks.
"""

from optparse import OptionParser
from SMART.Java.Python.structure.TranscriptContainer import TranscriptContainer
from commons.core.parsing.WigParser import WigParser
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc.RPlotter import RPlotter


if __name__ == "__main__":
    
    # parse command line
    description = "Get WIG Data v1.0.2: Compute the average data around some genomic coordinates using WIG files (thus covering a large proportion of the genome). [Category: WIG Tools]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",       dest="inputFileName",  action="store",                     type="string", help="input file [compulsory] [format: file in transcript format given by -f]")
    parser.add_option("-f", "--inputFormat", dest="inputFormat",    action="store",                     type="string", help="format of the input file [compulsory] [format: transcript file format]")
    parser.add_option("-w", "--wig",         dest="wig",            action="store",                     type="string", help="wig file name [compulsory] [format: file in WIG format]")    
    parser.add_option("-d", "--distance",    dest="distance",       action="store",      default=1000,  type="int",    help="distance around position [compulsory] [format: int] [default: 1000]")    
    parser.add_option("-s", "--strands",     dest="strands",        action="store_true", default=False,                help="consider both strands separately [format: boolean] [default: False]")    
    parser.add_option("-o", "--output",      dest="outputFileName", action="store",                     type="string", help="output file [compulsory] [format: output file in PNG format]")
    parser.add_option("-a", "--default",     dest="defaultValue",   action="store",      default=0.0,   type="float",  help="default value (when value is NA) [default: 0.0] [format: float]")
    parser.add_option("-l", "--log",         dest="log",            action="store_true", default=False,                help="use log scale for y-axis [format: boolean] [default: False]")
    parser.add_option("-k", "--keep",        dest="keep",           action="store_true", default=False,                help="keep temporary files [format: boolean] [default: False]")
    parser.add_option("-v", "--verbosity",   dest="verbosity",      action="store",      default=1,     type="int",    help="trace level [format: int]")
    (options, args) = parser.parse_args()

    # create parsers and writers
    transcriptParser = TranscriptContainer(options.inputFileName, options.inputFormat, options.verbosity)
    wigParser        = WigParser(options.wig)
    wigParser.setStrands(options.strands)
    wigParser.setDefaultValue(options.defaultValue)
    
    # allocate data
    strands = (1, -1) if options.strands else (1, )
    values    = {}
    for strand in strands:
        values[strand] = dict([(i, 0.0) for i in range(-options.distance, options.distance+1)])

    # read transcripts
    progress = Progress(transcriptParser.getNbTranscripts(), "Parsing %s" % (options.inputFileName), options.verbosity)
    for transcript in transcriptParser.getIterator():
        transcript.removeExons()
        transcript.restrictStart(2)
        transcript.extendStart(options.distance)
        transcript.extendEnd(options.distance-1)
        theseValues = transcript.extractWigData(wigParser)
        if len(strands) == 1:
            theseValues = {1: theseValues}
        for strand in strands:
            if len(theseValues[strand]) < 2 * options.distance + 1:
                theseValues[strand] = [options.defaultValue] * (2 * options.distance + 1 - len(theseValues[strand])) + theseValues[strand]
            if len(theseValues[strand]) != 2 * options.distance + 1:
				raise Exception("Got something wrong with the size of the WIG data concerning %s: %d found instead of %d" % (transcript, len(theseValues[strand]), 2 * options.distance + 1))
            for i in range(-options.distance, options.distance+1):
                values[strand][i] += theseValues[strand][i + options.distance]
        progress.inc()
    progress.done()

    for strand in strands:
        for i in range(-options.distance, options.distance+1):
            values[strand][i] /= transcriptParser.getNbTranscripts() * strand

    # draw plot
    plotter = RPlotter(options.outputFileName, options.verbosity, options.keep)
    plotter.setXLabel("Distance")
    plotter.setYLabel("WigValue")
    for strand in strands:
        plotter.addLine(values[strand])
    if options.log:
        plotter.setLog("y")
    plotter.plot()

