import re
import MySQLdb
from mySql.mySqlConnection import *


class MySqlTable(object):
  """
  Store a table of a mySQL database, used for transcripts or exons
  Record a a name and a type (int, float, double) for each column
  @ivar name:            name of the table
  @type name:            string
  @ivar variables:       name of the columns
  @type variables:       list of string
  @ivar types:           type of the columns
  @type types:           dict of string
  @ivar mySqlConnection: connection to a database
  @type mySqlConnection: class L{MySqlConnection<MySqlConnection>}
  @ivar nbLines:         number of rows
  @type nbLines:         int
  @ivar verbosity:       verbosity
  @type verbosity:       int
  """

  def __init__(self, name, verbosity = 0):
    """
    Constructor
    Possibly retrieve column names and types if table exists
    @param mySqlConnection: connection to a databas
    @type  mySqlConnection: class L{MySqlConnection<MySqlConnection>}
    @param name:            name of the table
    @type  name:            string
    @param verbosity:       verbosity
    @type  verbosity:       int
    """
    self.name            = name
    self.variables       = []
    self.types           = {}
    self.sizes           = {}
    self.nbLines         = None
    self.verbosity       = verbosity
    self.mySqlConnection = MySqlConnection(self.verbosity)
    queryTables = self.mySqlConnection.executeQuery("SHOW TABLES")
    for table in queryTables.getIterator():
      if table[0] == name:
        queryFields = self.mySqlConnection.executeQuery("DESCRIBE %s" % (name))
        for field in queryFields.getIterator():
          self.variables.append(field[0])
          m = re.search(r"(\w*)\((\d+)\)", field[1])
          if m == None:
            sys.exit("Error! MySQL format '%s' is wierd!" % (field[1]))
          self.types[field[0]] = m.group(1)
          self.sizes[field[0]] = int(m.group(2))


  def create(self, variables, types, sizes):
    """
    Create a table using give column names and types
    @param variables: names of the columns
    @type  variables: list of string
    @param types:     types of the columns
    @type  types:     dict of string
    @param sizes:     sizes of the types
    @type  sizes:     dict of int
    """
    self.variables = variables
    self.types     = types
    self.sizes     = sizes
    self.remove()
    query = "CREATE TABLE %s (id INT NOT NULL AUTO_INCREMENT" % (self.name)
    for variable in variables:
      query = "%s, %s %s(%d)" % (query, variable, types[variable], sizes[variable])
    query += ", PRIMARY KEY (id));"
    self.mySqlConnection.executeQuery(query)


  def loadFromFile(self, fileName):
    """
    Load the content from a file
    @param fileName: the name of the file
    @type  fileName: string
    """
    self.mySqlConnection.executeQuery("LOAD DATA LOCAL INFILE '%s' INTO TABLE %s" % (fileName, self.name))
    
    
  def rename(self, name):
    """
    Rename the table
    @param name: the new name
    @type  name: string
    """
    self.mySqlConnection.executeQuery("RENAME TABLE %s TO %s" % (self.name, name))
    self.name = name
    
  
  def copy(self, table):
    """
    Copy the given table this one
    @param table: the table to be copied
    @type  table: class L{MySqlTable<MySqlTable>}
    """
    self.mySqlConnection.executeQuery("CREATE TABLE %s LIKE %s" % (self.name, table.name))
    self.mySqlConnection.executeQuery("INSERT %s SELECT * FROM %s" % (self.name, table.name), True)
    

  def add(self, table):
    """
    Add the content of a table to this one
    @param table: the table to be added
    @type  table: class L{MySqlTable<MySqlTable>}
    """
    self.mySqlConnection.executeQuery("INSERT %s SELECT * FROM %s" % (self.name, table.name), True)
    
    
  def exists(self):
    """
    Check if the table exists in mySQL
    @return: true if it exits
    """
    query = self.mySqlConnection.executeQuery("SHOW TABLES LIKE '%s'" % (self.name))
    tables = query.getLines()
    return (tables != None) and (len(tables) > 0)
  

  def remove(self):
    """
    Remove this table
    """
    if self.exists():
      query = "DROP TABLE IF EXISTS %s" % (self.name)
      self.mySqlConnection.executeQuery(query)
    
    
  def clear(self):
    """
    Clear the content of this table
    """
    self.mySqlConnection.executeQuery("DELETE FROM %s" % (self.name))
    
    
  def getNbElements(self):
    """
    Count the number of rows in the table
    """
    command = "SELECT COUNT(*) FROM %s" % (self.name)
    query = self.mySqlConnection.executeQuery(command)
    return int(query.getLine()[0])


  def formatSql(self, value, type, size):
    """
    Format a value using MySQL encapsulation
    """
    if type.find("int") != -1:
      return "%d" % value
    if type.find("float") != -1:
      return "%.10f" % value
    if type.find("double") != -1:
      return "%.20f" % value
    if type.find("varchar") != -1:
      if len(value) > size:
        return "'%s'" % value[0:size]
      return "'%s'" % value
    sys.exit("Do not understand type %s" % (type))
  formatSql = classmethod(formatSql)


  def addLine(self, values):
    """
    Add a row to this table
    @param values: the values of the row
    @type  values: dict
    @return:       the id of the added row
    """
    sqlValues = []
    for variable in self.variables:
      sqlValues.append(self.formatSql(values[variable], self.types[variable], self.sizes[variable]))
    command = "INSERT INTO %s (%s) VALUES (%s)" % (self.name, ", ".join(self.variables), ", ".join(sqlValues))
    query   = self.mySqlConnection.executeQuery(command, True)
    id      = query.getInsertedId()
    return id
  
  
  def retrieveFromId(self, id):
    """
    Retrieve a row from its id
    @param id: the id of the row
    @type  id: int
    @return:   the row
    """
    query = self.mySqlConnection.executeQuery("SELECT * FROM %s WHERE id = %d" % (self.name, id))
    result = query.getLine()
    if result == None:
      sys.exit("Error! Id %d is not in the table %s!" % (id, self.name))
    return result


  def removeFromId(self, id):
    """
    Remove a row from its id
    @param id: the id of the row
    @type  id: int
    """
    self.mySqlConnection.executeQuery("DELETE FROM %s WHERE id = %d" % (self.name, id))
  
  
  def getIterator(self):
    """
    Iterate on the content of table
    @return: iterator to the rows of the table
    """
    query = self.mySqlConnection.executeQuery("SELECT * FROM %s" % (self.name))
    for line in query.getIterator():
      yield line


  def createIndex(self, indexName, values, unique = False, fullText = False):
    """
    Add an index on the table
    @param indexName: name of the index
    @type  indexName: string
    @param values:    values to be indexed
    @type  values:    string
    @param unique:    if the index is unique
    @type  unique:    boolean
    @param fullText:  whether full text should be indexed
    @type  fullText:  boolean
    """
    self.mySqlConnection.executeQuery("CREATE %s%sINDEX %s ON %s (%s)" % ("UNIQUE " if unique else "", "FULLTEXT " if fullText else "", indexName, self.name, ", ".join(values)))


  def setDefaultTagValue(self, field, name, value):
    """
    Add a tag value
    @param name:  name of the tag
    @type  name:  string
    @param value: value of the tag
    @type  value: string or int
    """
    for line in MySqlTable.getIterator(self):
      id   = line[0]
      tags = line[field]
      if tags == '':
        newTag = "%s=%s" % (name, value)
      else:
        newTag = "%s;%s=%s" % (tags, name, value)
      if name not in [tag.split("=")[0] for tag in tags.split(";")]:
        query = self.mySqlConnection.executeQuery("UPDATE %s SET tags = '%s' WHERE id = %i" % (self.name, newTag, id))


  def show(self):
    """
    Drop the content of the current table
    """
    query = self.mySqlConnection.executeQuery("SELECT * FROM %s" % (self.name))
    print query.getLines()


