import os
import random
from writer.bedWriter import *
from writer.ucscWriter import *
from writer.gbWriter import *
from writer.gff2Writer import *
from writer.gff3Writer import *
from writer.samWriter import *
from writer.wigWriter import *
from writer.mySqlTranscriptWriter import *
from misc.progress import *

class TranscriptWriter(object):
  """
  An interface class that writes a list of transcripts, handle different formats
  @ivar container:            container of the data
  @type container:            L{TranscriptContainer<TranscriptContainer>}
  @ivar format:               format of the data to be printed
  @type format:               string   
  @ivar file:                 the file where to print
  @type file:                 string 
  @ivar type:                 type of the data (transcripts, mappings or mySQL)
  @type type:                 string
  @ivar writer:               a transcript list writer
  @type writer:               L{TranscriptListWriter<TranscriptListWriter>} or None
  @ivar mode:                 use a container or enter transcript one by one
  @type mode:                 string
  @ivar verbosity:            verbosity
  @type verbosity:            int    
  """

  def __init__(self, file, format, verbosity = 0):
    """
    Constructor
    @param container: container of the data
    @type  container: string
    @param format:    format of the data
    @type  format:    string
    @param file:      file where to print
    @type file:       string
    @param verbosity: verbosity
    @type  verbosity: int
    """
    self.container            = None
    self.format               = format
    self.file                 = file
    self.verbosity            = verbosity
    self.type                 = None
    self.writer               = None
    self.mode                 = None
    if self.format == None:
      sys.exit("Error! Writer input format is empty!")
      
    if self.format == "bed":
      self.writer = BedWriter("%s.bed" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "ucsc":
      self.writer = UcscWriter("%s_ucsc.bed" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "gb":
      self.writer = GbWriter("%s.gb" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "gff2":
      self.writer = Gff2Writer("%s.gff2" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "gff3":
      self.writer = Gff3Writer("%s.gff3" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "gff":
      self.writer = Gff3Writer("%s.gff3" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "sam":
      self.writer = SamWriter("%s.sam" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "wig":
      self.writer = WigWriter("%s.wig" % (self.file), self.verbosity)
      self.type   = "transcript"
    elif self.format == "sql":
      self.type = "sql"
      pos = self.file.rfind(os.sep)
      if pos > -1:
        self.file = self.file[pos+1:]
      self.writer = MySqlTranscriptWriter(self.file, self.verbosity)
    else:
      sys.exit("Do not understand format '%s'" % (self.format))
      
      
  def setContainer(self, container):
    """
    Set a container for the data
    @param container: container of the data
    @type  container: class L{TranscriptContainer<TranscriptContainer>}
    """
    self.container = container
    if self.mode == "transcript":
      sys.exit("Error! TranscriptWriter '%s' on 'transcript' mode is currently used on 'container' mode." % (self.file))
    self.mode      = "container"


  def addTranscript(self, transcript):
    """
    Add a transcript to write
    @param transcript: a transcript
    @type transcript: class L{Transcript<Transcript>}
    """
    self.writer.addTranscript(transcript)
    if self.mode == "container":
      sys.exit("Error! TranscriptWriter '%s' on 'container' mode is currently used on 'transcript' mode." % (self.file))
    self.mode = "transcript"
    
    
  def addElement(self, transcript):
    """
    Same as addTranscript
    """
    self.addTranscript(transcript)
  

  def setTitle(self, title):
    """
    Possibly write a title for the list
    @param title: a title for the list
    @type title:  string
    """
    if self.writer != None:
      self.writer.setTitle(title)
      
    
  def write(self):
    """
    Write the content and possibly convert data
    """    
    if self.type == "transcript":
      if self.mode == "container":
        progress = Progress(self.container.getNbTranscripts(), "Writing transcripts", self.verbosity)
        for transcript in self.container.getIterator():
          self.writer.addTranscript(transcript)
          progress.inc()
        progress.done()
      return

    if self.mode == "transcript":
      self.writer.write()
      return

    if self.container.format != "sql":
      self.container.storeIntoDatabase()
    tables = self.container.getTables()
    for chromosome in tables:
      tables[chromosome].rename("%s_%s" % (self.file, chromosome))
    return
    
      
      
