/**
 *
 * Copyright INRA-URGI 2009-2010
 * 
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software. You can use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 * 
 * As a counterpart to the access to the source code and rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty and the software's author, the holder of the
 * economic rights, and the successive licensors have only limited
 * liability.
 * 
 * In this respect, the user's attention is drawn to the risks associated
 * with loading, using, modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean that it is complicated to manipulate, and that also
 * therefore means that it is reserved for developers and experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and, more generally, to use and operate it in the
 * same conditions as regards security.
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */
import java.util.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;
import javax.swing.*;
import javax.swing.filechooser.*;
import javax.swing.border.*;
import javax.swing.SwingUtilities;


public class ProgramOption {
  boolean       input;
  String        identifier;
  String        type;
  String        comment;
  boolean       compulsory;
  String[]      format;
  String        formatIdentifier;
  ProgramOption associatedOption;
  String        defaultValue;
  String[]      choices;
  JComponent    component;
  JPanel        panel;


  public ProgramOption() {
    this.input            = true;
    this.identifier       = null;
    this.type             = null;
    this.comment          = null;
    this.compulsory       = false;
    this.format           = null;
    this.formatIdentifier = null;
    this.associatedOption = null;
    this.defaultValue     = "";
    this.choices          = null;
    this.component        = null;
    this.panel            = null;
  }


  public void setInput(boolean input) {
    this.input = input;
  }


  public void setIdentifier(String identifier) {
    this.identifier = identifier;
  }


  public void setType(String type) {
    this.type = type;
  }


  public void setComment(String comment) {
    this.comment = comment;
  }


  public void setCompulsory(boolean compulsory) {
    this.compulsory = compulsory;
  }


  public void setFormat(String[] format) {
    this.format = format;
  }


  public void setFormat(String format) {
    this.format    = new String[1];
    this.format[0] = format;
  }


  public void setFormatIdentifier(String formatIdentifier) {
    this.formatIdentifier = formatIdentifier;
  }


  public void setAssociatedOption(ProgramOption option) {
    this.associatedOption = option;
  }


  public void setChoices(String[] choices) {
    this.choices = new String[choices.length+1];
    this.choices[0] = "---";
    for (int i = 0; i < choices.length; i++) {
      this.choices[i+1] = choices[i];
    }
  }


  public void setDefault(String defaultValue) {
    this.defaultValue = defaultValue;
  }


  public boolean isInput() {
    return this.input;
  }


  public boolean checkSettings() {
    if (this.identifier == null) {
      return false;
    }
    if (this.type == null) {
      return false;
    }
    if (this.comment == null) {
      return false;
    }
    if (this.comment == null) {
      return false;
    }
    if (("choice".compareToIgnoreCase(this.type) == 0) && (this.choices == null)) {
      return false;
    }
    return true;
  }


  public JPanel getPanel() {
    if (this.panel != null) {
      return this.panel;
    }
    String comment = this.comment;
    if (this.compulsory) {
      comment += " [*]";
    }

    GridLayout horizontalLayout = new GridLayout(1, 0);
    this.panel = new JPanel(false);
    this.panel.setLayout(horizontalLayout);
    JLabel label = new JLabel(comment);

    if (this.type == null) {
      System.out.println("Error! Option '" + this.identifier + "' is not set!");
    }

    if (("int".compareToIgnoreCase(this.type) == 0) || ("float".compareToIgnoreCase(this.type) == 0) || ("string".compareToIgnoreCase(this.type) == 0) || (("file".compareToIgnoreCase(this.type) == 0) && (!this.input))) {
      this.component = new JTextField();
      if (this.defaultValue != null) {
        ((JTextField) this.component).setText(this.defaultValue);
      }
      label.setLabelFor(this.component);
      this.panel.add(label);
      this.panel.add(this.component);
    }
    else if ("file".compareToIgnoreCase(this.type) == 0) {
      this.component = new JComboBox(Global.fileNames);
      label.setLabelFor(this.component);
      this.panel.add(label);
      this.panel.add(this.component);
    }
    else if ("boolean".compareToIgnoreCase(this.type) == 0) {
      this.component = new JCheckBox();
      if ((this.defaultValue != null) && (this.defaultValue.compareToIgnoreCase("true") == 0)) {
        ((JCheckBox) this.component).setSelected(true);
      }
      label.setLabelFor(this.component);
      this.panel.add(label);
      this.panel.add(this.component);
    }
    else if ("format".compareToIgnoreCase(this.type) == 0) {
      Vector < String > formats = new Vector < String > ();
      for (String format: this.format) {
        if (Global.formats.getFormats(format) == null) {
          System.out.println("Do not know how to handle format '" + format + "'.");
        }
        formats.addAll(Global.formats.getFormats(format).getFormats());
      }
      this.component = new JComboBox(formats);
      label.setLabelFor(this.component);
      this.panel.add(label);
      this.panel.add(this.component);
    }
    else if ("files".compareToIgnoreCase(this.type) == 0) {
      JButton button = new JButton("file...");
      this.component = new JTextField();
      label.setLabelFor(this.component);
      this.panel.add(label);
      this.panel.add(this.component);
      this.panel.add(button);
      Global.otherFileConcatenationChooser.put(button, (JTextField) this.component);
    }
    else if ("directory".compareToIgnoreCase(this.type) == 0) {
      JButton button = new JButton("directory...");
      this.component = new JTextField();
      label.setLabelFor(this.component);
      this.panel.add(label);
      JPanel rightPanel = new JPanel(false);
      rightPanel.setLayout(new BoxLayout(rightPanel, BoxLayout.LINE_AXIS));
      rightPanel.add(this.component);
      rightPanel.add(button);
      this.panel.add(rightPanel);
      Global.otherDirectoriesChooser.put(button, (JTextField) this.component);
    }
    else if ("choice".compareToIgnoreCase(this.type) == 0) {
      this.component = new JComboBox(this.choices);
      label.setLabelFor(this.component);
      this.panel.add(label);
      this.panel.add(this.component);
    }
    else {
      System.out.println("Do not know how to read type " + this.type);
    }

    return this.panel;
  }


  public JComponent getComponent() {
    if (component == null) {
      this.getPanel();
    }
    return this.component;
  }


  private String getValue() {
    if (("int".equals(this.type)) || ("float".equals(this.type)) || ("string".equals(this.type)) || (("file".equals(this.type)) && (! this.input)) || ("directory".equals(this.type)) || ("files".equals(this.type)))  {
      String s = ((JTextField) this.component).getText();
      if ("None".equals(s)) {
        return "";
      }
      return s;
    }
    if ("file".equals(this.type)) {
      return (String) ((JComboBox) this.component).getSelectedItem();
    }
    if ("boolean".equals(this.type)) {
      return ((JCheckBox) this.component).isSelected()? "true": "false";
    }
    if ("format".equals(this.type)) {
      return (String) ((JComboBox) this.component).getSelectedItem();
    }
    if ("choice".equals(this.type)) {
      String s = (String) ((JComboBox) this.component).getSelectedItem();
      if ("---".equals(s)) {
        return "";
      }
      return s;
    }
    System.out.println("Do not know how to get value of '" + this.type + "' (" + this.identifier + ").");
    return null;
  }


  public String checkValue() {
    String value = this.getValue();
    if ((this.compulsory) && ((value == null) || ("".equals(value)))) {
      return "Option '" + this.comment + "' has no value... Please specify it.\n";
    }
    if ("int".equals(this.type)) {
      if ((value != null) && (! "".equals(value)) && (! "None".equals(value))) {
        try {
          int i = Integer.parseInt(value);
        }
        catch (NumberFormatException e) {
          return "Option '" + this.comment + "' should be an integer... Please correct it.\n";
        }
      }
    }
    else if ("float".equals(this.type)) {
      if ((value != null) && (! "".equals(value))) {
        try {
          float i = Float.parseFloat(value);
        }
        catch (NumberFormatException e) {
          return "Option '" + this.comment + "' should be a float... Please correct it.\n";
        }
      }
    }
    return null;
  }


  public LinkedList <String> getCommand() {
    LinkedList <String> list = new LinkedList <String> ();

    if (("int".equals(this.type)) || ("float".equals(this.type)) || ("string".equals(this.type)) || (("file".equals(this.type)) && (! this.input)) || ("format".equals(this.type)) || ("directory".equals(this.type)) || ("files".equals(this.type)) || ("choice".equals(this.type))) {
      String value = this.getValue();
      if (value.length() == 0) {
        return list;
      }
      list.add(this.identifier);
      list.add(value);
      return list;
    }
    if ("file".equals(this.type)) {
      String fileName = (String) ((JComboBox) this.component).getSelectedItem();
      if (fileName == null) {
        return list;
      }
      list.add(this.identifier);
      list.add(this.getValue());
      return list;
    }
    if (("boolean".equals(this.type)) || ("bool".equals(this.type))) {
      if ("true".equals(this.getValue())) {
        list.add(this.identifier);
      }
      return list;
    }
    System.out.println("Cannot get type of option " + this.type + " (" + this.identifier + "): " + this.getValue());
    return null;
  }


  public File getOutputFile() {
    if (this.input) return null;
    String format = "";
    if (this.format != null) {
      format = this.format[0];
    }
    if (this.associatedOption != null) {
      format = this.associatedOption.getValue();
    }
    return new File(this.getValue(), Global.formats.getFormatType(format), format);
  }
}
