#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""
Remove empty sequences from a FASTA or FASTQ file
"""

import os, random
from optparse import OptionParser
from commons.core.parsing.FastaParser import *
from commons.core.parsing.FastqParser import *
from commons.core.writer.FastaWriter import *
from commons.core.writer.FastqWriter import *
from SMART.Java.Python.misc.Progress import *


class EmptySequenceRemover(object):

    def __init__(self, verbosity = 1):
        self.verbosity            = verbosity
        self.inputFileName    = None
        self.parser                 = None
        self.format                 = None
        self.writer                 = None
        self.forbiddenNames = {}
        self.removedNames     = {}


    def setInputFileName(self, fileName, format):
        self.inputFileName = fileName
        self.format                = format
        if options.format == "fasta":
            self.parser = FastaParser(self.inputFileName, self.verbosity)
        elif options.format == "fastq":
            self.parser = FastqParser(self.inputFileName, self.verbosity)
        else:
            sys.exit("Do not understand '%s' file format." % (self.format))


    def setOutputFileName(self, fileName):
        if options.format == "fasta":
            self.writer = FastaWriter(fileName, self.verbosity)
        elif options.format == "fastq":
            self.writer = FastqWriter(fileName, self.verbosity)


    def parse(self):
        progress = Progress(self.parser.getNbSequences(), "Reading sequences in %s" % (options.inputFileName), options.verbosity)
        for sequence in self.parser.getIterator():
            name = sequence.name.split("/")[0]
            if name not in self.forbiddenNames:
                if sequence.sequence == "":
                    self.removedNames[name] = 1
                else:
                    self.writer.addSequence(sequence)
            progress.inc()
        progress.done()
        self.writer.write()


if __name__ == "__main__":
    
    # parse command line
    description = "Remove Empty Sequences v1.0.2: Remove all the empty sequences in a list. [Category: Personal]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",     dest="inputFileName",   action="store",                     type="string", help="input file [compulsory] [format: file in sequence format given by -f]")
    parser.add_option("-f", "--format",    dest="format",          action="store",                     type="string", help="format of the input file [compulsory] [format: sequence file format]")
    parser.add_option("-j", "--input2",    dest="inputFileName2",  action="store",                     type="string", help="input file 2 (in case of pair end reads) [format: file in sequence format given by -f] [default: None]")
    parser.add_option("-o", "--output",    dest="outputFileName",  action="store",      default=None,  type="string", help="output file [compulsory] [format: output file in format given by -f]")
    parser.add_option("-p", "--output2",   dest="outputFileName2", action="store",      default=None,  type="string", help="output file 2 (in case of pair end reads) [format: output file in sequence format given by -f] [default: None]")
    parser.add_option("-v", "--verbosity", dest="verbosity",       action="store",      default=1,     type="int",    help="trace level [format: int] [default: 1]")
    parser.add_option("-l", "--log",       dest="log",             action="store_true", default=False,                help="write a log file [format: bool] [default: false]")
    (options, args) = parser.parse_args()

    if options.log:
        logHandle = open("%s.log" % options.outputFileName, "w")
    
    remover = EmptySequenceRemover(options.verbosity)
    remover.setInputFileName(options.inputFileName, options.format)
    remover.setOutputFileName(options.outputFileName)
    remover.parse()
    removedNames = remover.removedNames
    if options.log:
        for name in removedNames:
            logHandle.write("Removed '%s' in %s\n" % (name, options.inputFileName))
    nbSequences = remover.parser.getNbSequences()

    newRemovedNames = {}
    if options.inputFileName2 != None:
        remover = EmptySequenceRemover(options.verbosity)
        remover.setInputFileName(options.inputFileName2, options.format)
        remover.setOutputFileName(options.outputFileName2)
        remover.forbiddenNames = removedNames
        remover.parse()
        newRemovedNames = remover.removedNames
        if options.log:
            for name in newRemovedNames:
                logHandle.write("Removed '%s' in %s\n" % (name, options.inputFileName2))

        remover = EmptySequenceRemover(options.verbosity)
        remover.setInputFileName(options.inputFileName, options.format)
        remover.setOutputFileName(options.outputFileName)
        remover.forbiddenNames = newRemovedNames
        remover.parse()

    nbRemoved = len(removedNames.keys()) + len(newRemovedNames.keys())
    print "%d over %d sequences are empty (%.2f%%)." % (nbRemoved, nbSequences, float(nbRemoved) / nbSequences * 100)
