#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2011
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import random, os, os.path, time, sqlite3
from optparse import OptionParser
from commons.core.parsing.ParserChooser import ParserChooser
from commons.core.writer.TranscriptWriter import TranscriptWriter
from SMART.Java.Python.structure.Interval import Interval
from SMART.Java.Python.structure.Transcript import Transcript
from SMART.Java.Python.structure.Mapping import Mapping
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc.UnlimitedProgress import UnlimitedProgress
try:
   import cPickle as pickle
except:
   import pickle

MINBIN = 3
MAXBIN = 7


def getBin(start, end):
	for i in range(MINBIN, MAXBIN + 1):
		binLevel = 10 ** i
		if int(start / binLevel) == int(end / binLevel):
			return int(i * 10 ** (MAXBIN + 1) + int(start / binLevel))
	return int((MAXBIN + 1) * 10 ** (MAXBIN + 1))

def getOverlappingBins(start, end):
	array	= []
	bigBin = int((MAXBIN + 1) * 10 ** (MAXBIN + 1))
	for i in range(MINBIN, MAXBIN + 1):
		binLevel = 10 ** i
		array.append((int(i * 10 ** (MAXBIN + 1) + int(start / binLevel)), int(i * 10 ** (MAXBIN + 1) + int(end / binLevel))))
	array.append((bigBin, bigBin))
	return array


class FindOverlapsWithSeveralIntervalsBin(object):

	def __init__(self, verbosity):
		self.verbosity	= verbosity
		self.randomNumber = random.randint(0, 10000)
		self.dbName	   = "smartdb%d" % (self.randomNumber)
		if "SMARTTMPPATH" in os.environ:
			self.dbName = os.join(os.environ["SMARTTMPPATH"], self.dbName)
		self.connection = sqlite3.connect(self.dbName)
		self.tableNames = {}
		self.nbQueries  = 0
		self.nbRefs	 = 0
		self.nbWritten  = 0
		self.nbOverlaps = 0
		cursor = self.connection.cursor()
		cursor.execute("PRAGMA journal_mode = OFF")
		cursor.execute("PRAGMA synchronous = 0")
		cursor.execute("PRAGMA locking_mode = EXCLUSIVE")
		cursor.execute("PRAGMA count_change = OFF")
		cursor.execute("PRAGMA temp_store = 2")

	def __del__(self):
		cursor = self.connection.cursor()
		for tableName in self.tableNames.values():
			cursor.execute("DROP TABLE IF EXISTS %s" % (tableName))
		if os.path.exists(self.dbName):
			os.remove(self.dbName)
		
	def createTable(self, chromosome):
		cursor = self.connection.cursor()
		tableName = "tmpTable_%s_%d" % (chromosome.replace("-", "_"), self.randomNumber)
		cursor.execute("CREATE TABLE %s (start INT, end INT, transcript BLOB, bin INT)" % (tableName))
		cursor.execute("CREATE INDEX index_%s ON %s (bin)" % (tableName, tableName))
		self.tableNames[chromosome] = tableName

	def setReferenceFile(self, fileName, format):
		chooser = ParserChooser(self.verbosity)
		chooser.findFormat(format)
		parser = chooser.getParser(fileName)
		startTime = time.time()
		if self.verbosity > 2:
			print "Storing into table"
		for transcript in parser.getIterator():
			if transcript.__class__.__name__ == "Mapping":
				transcript = transcript.getTranscript()
			transcriptString = pickle.dumps(transcript)
			chromosome = transcript.getChromosome()
			if chromosome not in self.tableNames:
				self.createTable(chromosome)
			start	  = transcript.getStart()
			end		= transcript.getEnd()
			bin		= getBin(start, end)
			cursor	 = self.connection.cursor()
			cursor.execute("INSERT INTO %s (start, end, transcript, bin) VALUES (?, ?, ?, ?)" % (self.tableNames[chromosome]), (start, end, sqlite3.Binary(transcriptString), bin))
			self.nbRefs += 1
		self.connection.commit()
		endTime = time.time()
		if self.verbosity > 2:
			print "	...done (%.2gs)" % (endTime - startTime)

	def setQueryFile(self, fileName, format):
		chooser = ParserChooser(self.verbosity)
		chooser.findFormat(format)
		self.queryParser = chooser.getParser(fileName)
		self.nbQueries = self.queryParser.getNbItems()

	def setOutputFile(self, fileName):
		self.writer = TranscriptWriter(fileName, "gff3", self.verbosity)

	def compare(self):
		progress = Progress(self.nbQueries, "Reading queries", self.verbosity)
		startTime = time.time()
		for queryTranscript in self.queryParser.getIterator():
			if queryTranscript.__class__.__name__ == "Mapping":
				queryTranscript = queryTranscript.getTranscript()
			progress.inc()
			queryChromosome = queryTranscript.getChromosome()
			if queryChromosome not in self.tableNames:
				continue
			queryStart = queryTranscript.getStart()
			queryEnd   = queryTranscript.getEnd()
			bins	   = getOverlappingBins(queryStart, queryEnd)
			commands   = []
			for bin in bins:
				command = "SELECT * FROM %s WHERE bin " % (self.tableNames[queryChromosome])
				if bin[0] == bin[1]:
					command += "= %d" % (bin[0])
				else:
					command += "BETWEEN %d AND %d" % (bin[0], bin[1])
				commands.append(command)
			command = " UNION ".join(commands)
			cursor  = self.connection.cursor()
			cursor.execute(command)
			overlap = False
			line	= cursor.fetchone()
			while line:
				refStart, refEnd, refTranscriptString, refBin = line
				if refStart <= queryEnd and refEnd >= queryStart:
					refTranscript = pickle.loads(str(refTranscriptString))
					if refTranscript.overlapWith(queryTranscript):
						overlap = True
						self.nbOverlaps += 1
				line = cursor.fetchone()
			if overlap:
				self.writer.addTranscript(queryTranscript)
				self.nbWritten += 1
		progress.done()
		endTime = time.time()
		self.timeSpent = endTime - startTime

	def displayResults(self):
		print "# queries:  %d" % (self.nbQueries)
		print "# refs:	   %d" % (self.nbRefs)
		print "# written:  %d (%d overlaps)" % (self.nbWritten, self.nbOverlaps)
		print "time:	   %.2gs" % (self.timeSpent)

	def run(self):
		self.compare()
		self.displayResults()

if __name__ == "__main__":
	
	description = "Find Overlaps With Several Intervals Using Bin v1.0.1: Use MySQL binning to compare intervals. [Category: Personal]"

	parser = OptionParser(description = description)
	parser.add_option("-i", "--input1",	  dest="inputFileName1", action="store",			type="string", help="query input file [compulsory] [format: file in transcript format given by -f]")
	parser.add_option("-f", "--format1",	 dest="format1",		action="store",			type="string", help="format of previous file [compulsory] [format: transcript file format]")
	parser.add_option("-j", "--input2",	  dest="inputFileName2", action="store",			type="string", help="reference input file [compulsory] [format: file in transcript format given by -g]")
	parser.add_option("-g", "--format2",	 dest="format2",		action="store",			type="string", help="format of previous file [compulsory] [format: transcript file format]")
	parser.add_option("-o", "--output",	  dest="outputFileName", action="store",			type="string", help="output file [format: output file in GFF3 format]")
	parser.add_option("-v", "--verbosity",   dest="verbosity",	  action="store", default=1, type="int",	help="trace level [format: int]")
	(options, args) = parser.parse_args()

	fowsib = FindOverlapsWithSeveralIntervalsBin(options.verbosity)
	fowsib.setQueryFile(options.inputFileName1, options.format1)
	fowsib.setReferenceFile(options.inputFileName2, options.format2)
	fowsib.setOutputFile(options.outputFileName)
	fowsib.run()

