#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
from SMART.Java.Python.structure.SubMapping import SubMapping
from SMART.Java.Python.structure.Transcript import Transcript
from SMART.Java.Python.structure.Interval import Interval

class Mapping(object):
    """A class that represents a mapping"""

    def __init__(self):
        self.targetInterval = None
        self.queryInterval  = None
        self.subMappings    = []
        self.size           = None
        self.transcript     = None
        self.tags           = {}


    def copy(self, mapping):
        for subMapping in mapping.subMappings:
            newSubMapping = SubMapping(subMapping)
            self.addSubMapping(newSubMapping)
        self.targetInterval = Interval(mapping.targetInterval)
        self.queryInterval  = Interval(mapping.queryInterval)
        self.size           = mapping.size
        self.tags           = {}
        for tag in mapping.tags:
            self.tags[tag] = mapping[tag]
        self.transcript.copy(mapping.transcript)


    def setTargetInterval(self, interval):
        self.targetInterval = Interval(interval)
        if self.queryInterval != None:
            self.setDirection(self.targetInterval.getDirection() * self.queryInterval.getDirection())


    def setQueryInterval(self, interval):
        self.queryInterval = Interval(interval)
        if self.targetInterval != None:
            self.setDirection(self.targetInterval.getDirection() * self.queryInterval.getDirection())


    def getQueryInterval(self):
        return self.queryInterval


    def addSubMapping(self, subMapping):
        subMappingCopy = SubMapping(subMapping)
        self.subMappings.append(subMappingCopy)

        if self.targetInterval:
            self.targetInterval.setStart(min(self.targetInterval.getStart(), subMapping.targetInterval.getStart()))
            self.targetInterval.setEnd(max(self.targetInterval.getEnd(),     subMapping.targetInterval.getEnd()))
        else:
            self.setTargetInterval(subMapping.targetInterval)
        if self.queryInterval:
            self.queryInterval.setStart(min(self.queryInterval.getStart(), subMapping.queryInterval.getStart()))
            self.queryInterval.setEnd(max(self.queryInterval.getEnd(),     subMapping.queryInterval.getEnd()))
        else:
            self.setQueryInterval(subMapping.queryInterval)

        if self.getDirection() != 0:
            subMapping.setDirection(self.getDirection())
        if self.size == None:
            self.size = 0
        if "identity" in subMapping.getTagNames() and "identity" not in self.getTagNames():
            self.setTagValue("identity", subMapping.getTagValue("identity"))
        elif "identity" in subMapping.getTagNames() and subMapping.size != None:
            self.setTagValue("identity", (self.getTagValue("identity") * self.size + subMapping.getTagValue("identity") * subMapping.size) / (self.size + subMapping.size))
        if subMapping.size != None:
            self.size += subMapping.size
        if "nbMismatches" in subMapping.getTagNames() and "nbMismatches" not in self.getTagNames():
            self.setTagValue("nbMismatches", subMapping.getTagValue("nbMismatches"))
        elif "nbMismatches" in subMapping.getTagNames():
            self.setTagValue("nbMismatches", self.getTagValue("nbMismatches") + subMapping.getTagValue("nbMismatches"))
        if "nbGaps" in subMapping.getTagNames() and "nbGaps" not in self.getTagNames():
            self.setTagValue("nbGaps", subMapping.getTagValue("nbGaps"))
        elif "nbGaps" in subMapping.getTagNames():
            self.setTagValue("nbGaps", self.getTagValue("nbGaps") + subMapping.getTagValue("nbGaps"))


    def setDirection(self, direction):
        for subMapping in self.subMappings:
            subMapping.setDirection(direction)


    def getDirection(self):
        if not self.subMappings:
            raise Exception("Error! Mapping '%s' has no submapping" % (self))
        return self.subMappings[0].getDirection()


    def setSize(self, size):
        self.size = size
        if "identity" in self.getTagNames():
            self.setTagValue("nbMismatches", self.size - round(self.size * self.getTagValue("identity") / 100.0))


    def setTagValue(self, name, value):
        self.tags[name] = value
        self.transcript = None


    def getTagValue(self, name):
        return self.tags[name]

    
    def getTagNames(self):
        return self.tags.keys()


    def setIdentity(self, identity):
        self.setTagValue("identity", identity)
        if self.size != None and "nbMismatches" not in self.getTagNames():
            nbMismatches = 0 if self.size == 0 else self.size - round(self.size * self.getTagValue("identity") / 100.0)
            self.setTagValue("nbMismatches", nbMismatches)


    def setNbOccurrences(self, nbOccurrences):
        self.setTagValue("nbOccurrences", nbOccurrences)


    def setNbMismatches(self, nbMismatches):
        self.setTagValue("nbMismatches", nbMismatches)
        if self.size != None and "identity" not in self.getTagNames():
            identity = 100 if self.size == 0 else (self.size - self.getTagValue("nbMismatches")) / float(self.size) * 100
            self.setTagValue("identity", identity)


    def setNbGaps(self, nbGaps):
        self.setTagValue("nbGaps", nbGaps)
        
        
    def setRank(self, rank):
        self.setTagValue("rank", rank)
        

    def setEvalue(self, evalue):
        self.setTagValue("evalue", evalue)
        

    def setOccurrence(self, occurrence):
        self.setTagValue("occurrence", occurrence)
        
        
    def setBestRegion(self, bestRegion):
        self.setTagValue("bestRegion", bestRegion)


    def mergeExons(self, distance):
        previousSubMapping = None
        subMappings        = []
        for subMapping in self.subMappings:
            if previousSubMapping == None:
                subMappings.append(subMapping)
                previousSubMapping = subMapping
            else:
                targetDistance = subMapping.targetInterval.getDistance(previousSubMapping.targetInterval)
                queryDistance  = subMapping.queryInterval.getDistance(previousSubMapping.queryInterval)
                if targetDistance <= distance:
                    self.setTagValue("nbGaps", self.getTagValue("nbGaps") + queryDistance)
                    previousSubMapping.merge(subMapping)
                else:
                    subMappings.append(subMapping)
                    previousSubMapping = subMapping
        self.subMappings = subMappings
        
        
    def getTranscript(self):
        """
        Extract a transcript from this mapping
        @return: a transcript
        """
        if self.transcript != None:
            return self.transcript
        self.transcript = Transcript()
        self.transcript.copy(self.targetInterval)
        self.transcript.setDirection(self.getDirection())
        self.transcript.setName(self.queryInterval.getName())
        self.transcript.removeExons()
        if len(self.subMappings) > 1:
            for subMapping in self.subMappings:
                self.transcript.addExon(subMapping.targetInterval)
        cpt = 1
        for exon in self.transcript.exons:
            exon.setDirection(self.transcript.getDirection())
            exon.setName("%s-exon%d" % (self.transcript.getName(), cpt))
            exon.setChromosome(self.transcript.getChromosome())
            cpt += 1
        self.transcript.setDirection(self.getDirection())
        self.transcript.sortExons()
        for tag in self.tags:
            if "bestRegion" not in self.getTagNames():
                self.transcript.setTagValue("bestRegion", "(self)")
            self.transcript.setTagValue(tag, self.getTagValue(tag))
        return self.transcript
    

    def getChromosome(self):
        if not self.subMappings:
            raise Exception("Error! Mapping '%s' has no submapping" % (self))
        return self.subMappings[0].targetInterval.getChromosome()


    
    def getErrorScore(self):
        return self.getTagValue("nbGaps") * 3 + self.getTagValue("nbMismatches") + (len(self.subMappings) - 1) * 0.1
            

    def printGBrowseReference(self):
        return self.getTranscript().printGBrowseReference()


    def printGBrowseLine(self):
        return self.getTranscript().printGBrowseLine()


    def printGBrowse(self):
        return self.getTranscript().printGBrowse()


    def printBed(self):
        return self.getTranscript().printBed()


    def __str__(self):
        return "%s     ----     %s" % (str(self.getTranscript()), ", ". join([str(submapping) for submapping in self.subMappings]))
