import os
import random
from structure.transcript import *
from parsing.parserChooser import *
# from parsing.bedParser import *
# from parsing.gffParser import *
from mySql.mySqlConnection import *
from mySql.mySqlTable import *
from mySql.mySqlTranscriptTable import *
# from parsing.mapperParser import *
# from parsing.nucmerParser import *
# from parsing.seqmapParser import *
# from parsing.soapParser import *
# from parsing.blastParser import *
# from parsing.pslParser import *
# from parsing.rmapParser import *
# from parsing.shrimpParser import *
# from parsing.axtParser import *
# from parsing.exoParser import *
# from parsing.maqParser import *
# from parsing.samParser import *
from misc.progress import *

class TranscriptContainer(object):
  """
  An interface class that contains a list of transcripts, handle different formats
  @ivar container:            container of the data
  @type container:            string 
  @ivar format:               format of the data
  @type format:               string    
  @ivar transcriptListParser: possibly contains a parser to a list of transcripts
  @type transcriptListParser: L{TranscriptListParser<TranscriptListParser>} or None
  @ivar mappingListParser:    possibly contains a parser to a list of mappings
  @type mappingListParser:    L{MapperParser<MapperParser>} or None
  @ivar transcriptTables:     possibly contains the mySQL tables
  @type transcriptTables:     dict of L{MySqlTranscriptTable<MySqlTranscriptTable>} or None
  @ivar mySqlConnection:      connection to a MySQL database
  @type mySqlConnection:      class L{MySqlConnection<MySqlConnection>}
  @ivar type:                 type of the data (transcripts, mappings or mySQL)
  @type type:                 string
  @ivar verbosity:            verbosity
  @type verbosity:            int    
  """

  def __init__(self, container, format, verbosity = 0):
    """
    Constructor
    @param container: container of the data
    @type  container: string
    @param format:    format of the data
    @type  format:    string
    @param verbosity: verbosity
    @type  verbosity: int
    """
    self.container            = container
    self.format               = format
    self.verbosity            = verbosity
    self.transcriptListParser = None
    self.mappingListParser    = None
    self.transcriptTables     = {}
    self.mySqlConnection      = MySqlConnection(self.verbosity)
    self.foundData            = False
    self.nbTranscripts        = None
    self.nbNucleotides        = None
    self.chromosomes          = None
    self.type                 = None
    if self.container == None:
      sys.exit("Error! Container input file name is empty!")
    if self.format == None:
      sys.exit("Error! Container input format is empty!")
    
    
  def findData(self):
    """
    Load data
    """
    if self.format == None:
      sys.exit("Error! Format is not specified!")
    if self.format == "sql":
      self.transcriptTables = {}
      self.chromosomes      = []
      self.nbTranscripts    = 0
      self.nbNucleotides    = 0
      self.type             = "sql"
      query = self.mySqlConnection.executeQuery("SHOW TABLES LIKE '%s_%%_transcripts'" % (self.container))
      for line in query.getIterator():
        tableName = line[0]
        m = re.search(r"^(\S*)_transcripts$", tableName[len(self.container)+1:])
        if m == None:
          sys.exit("Table '%s' has a strange name" % (tableName))
        chromosome = m.group(1)
        self.transcriptTables[chromosome] = MySqlTranscriptTable(self.container, chromosome, self.verbosity)
        self.chromosomes.append(chromosome)
        for transcript in self.transcriptTables[chromosome].getIterator():
          self.nbTranscripts += 1
          self.nbNucleotides += transcript.getSize()
    if self.type == None:
      parserChooser = ParserChooser(self.verbosity)
      parserChooser.findFormat(self.format)
      self.type = parserChooser.getType()
      if self.type == "transcript":
        self.transcriptListParser = parserChooser.getParser(self.container)
      elif self.type == "mapping":
        self.mappingListParser = parserChooser.getParser(self.container)
      else:
        sys.exit("Error! Cannot handle format '%s'!" % (self.format))
#   if self.type == None:
#     for parserClass in TranscriptListParser.__subclasses__():
#       if self.format in parserClass.getFileFormats():
#         self.transcriptListParser = parserClass(self.container, self.verbosity)
#         self.type                 = "transcript"
#   if self.type == None:
#     for parserClass in MapperParser.__subclasses__():
#       if self.format in parserClass.getFileFormats():
#         self.mappingListParser = parserClass(self.container, self.verbosity)
#         self.type                 = "mapping"
    if self.type == None:
      sys.exit("Error! Cannot handle format '%s'!" % (self.format))

#   if self.format == "bed":
#     self.transcriptListParser = BedParser(self.container, self.verbosity)
#     self.type                 = "transcript"
#   elif self.format == "gff" or self.format == "gff2" or self.format == "gff3":
#     self.transcriptListParser = GffParser(self.container, self.verbosity)
#     self.type                 = "transcript"
#   elif self.format == "nucmer":
#     self.mappingListParser = NucmerParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "seqmap":
#     self.mappingListParser = SeqmapParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "soap":
#     self.mappingListParser = SoapParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "blast":
#     self.mappingListParser = BlastParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "psl":
#     self.mappingListParser = PslParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "rmap":
#     self.mappingListParser = RmapParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "shrimp":
#     self.mappingListParser = ShrimpParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "axt":
#     self.mappingListParser = AxtParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "exo":
#     self.mappingListParser = ExoParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "maq":
#     self.mappingListParser = MaqParser(self.container, self.verbosity)
#     self.type              = "mapping"
#   elif self.format == "sam":
#     self.mappingListParser = SamParser(self.container, self.verbosity)
#     self.type              = "mapping"
  
    if self.transcriptListParser != None:
      if self.type == "transcript":
        self.nbTranscripts = self.transcriptListParser.getNbTranscripts()
        self.nbNucleotides = self.transcriptListParser.getNbNucleotides()
    if self.mappingListParser != None:
      if self.type == "mapping":
        self.nbTranscripts = self.mappingListParser.getNbMappings()
        self.nbNucleotides = self.mappingListParser.getNbNucleotides()

    self.foundData = True


  def getNbTranscripts(self):
    """
    Get the number of transcripts
    @return: the number of transcripts
    """
    if not self.foundData:
      self.findData()
    return self.nbTranscripts
  
  
  def getNbItems(self):
    """
    Same as getNbTranscripts
    """
    return self.getNbTranscripts()


  def getNbNucleotides(self):
    """
    Get the number of nucleotides
    @return: the number of nucleotides
    """
    if not self.foundData:
      self.findData()
    return self.nbNucleotides


  def getChromosomes(self):
    """
    Get the chromosomes
    @return: the chromosomes
    """
    if not self.foundData:
      self.findData()
    return self.chromosomes
  

  def getIterator(self):
    """
    An iterator
    @return: an iterator to a list of transcripts
    """
    if not self.foundData:
      self.findData()
    if self.type == "sql":
      for chromosome in self.transcriptTables:
        for transcript in self.transcriptTables[chromosome].getIterator():
          yield transcript
      return
    if self.type == "transcript":
      for transcript in self.transcriptListParser.getIterator():
        yield transcript
      return
    if self.type == "mapping":
      for mapping in self.mappingListParser.getIterator():
        yield mapping.getTranscript()
      return
    sys.exit("Error! No valid transcript container given!")
    
    
  def storeIntoDatabase(self, name = None):
    """
    Store the current transcript / mapping list into database
    """
    if not self.foundData:
      self.findData()

    if self.transcriptListParser == None or len(self.transcriptTables.keys()) != 0:
      return
    
    mySqlTranscriptWriter = MySqlTranscriptWriter(name, self.verbosity)
    mySqlTranscriptWriter.addTranscriptList(self.transcriptListParser)
    mySqlTranscriptWriter.write()
    self.transcriptTables = mySqlTranscriptWriter.getTables()
    self.type = "sql"
      
      
  def getTables(self):
    """
    Accessor to the mySQL tables
    @return: the mySQL tables
    """
    return self.transcriptTables
    

  def setDefaultTagValue(self, name, value):
    """
    Set the given tag to the value for all transcripts
    @param name:  name of the tag
    @type  name:  string
    @param value: value of the tag
    @type  value: string
    """
    if self.type == "sql":
      for chromosome in self.transcriptTables:
        self.transcriptTables[chromosome].setDefaultTagValue(name, value)
    elif self.type == "transcript":
      self.transcriptListParser.setDefaultTagValue(name, value)
    elif self.type == "mapping":
      self.mappingListParser.setDefaultTagValue(name, value)

