#! /usr/bin/env python
"""
Plot the data from the data files
"""

import os
import re
from optparse import OptionParser
from parsing.gffParser import *
from misc.rPlotter import *
from misc.progress import *


if __name__ == "__main__":
  
  # parse command line
  description = "Plot Repartition: Plot the repartition of different data on a whole genome. (This tool uses 1 input file only, the different values being stored in the tags.  See documentation to know more about it.) [Category: Visualization]"

  parser = OptionParser(description = description)
  parser.add_option("-i", "--input",        dest="inputFileName",     action="store",                        type="string", help="input file name [compulsory] [format: file in GFF3 format]")
  parser.add_option("-n", "--names",        dest="names",              action="store",                       type="string", help="name for the tags (separated by commas and no space) [compulsory] [format: string]")
  parser.add_option("-o", "--output",       dest="outputFileName",    action="store",                        type="string", help="output file [compulsory] [format: output file in PNG format]")
  parser.add_option("-c", "--color",        dest="colors",            action="store",      default=None,     type="string", help="color of the lines (separated by commas and no space) [format: string]")
  parser.add_option("-f", "--format",       dest="format",            action="store",      default="png",    type="string", help="format of the output file [format: string] [default: png]")
  parser.add_option("-r", "--normalize",    dest="normalize",         action="store_true", default=False,                   help="normalize data (when panels are different) [format: bool] [default: false]")
  parser.add_option("-l", "--log",          dest="log",               action="store",      default="",       type="string", help="use log on x- or y-axis (write 'x', 'y' or 'xy') [format: string]")
  parser.add_option("-v", "--verbosity",    dest="verbosity",         action="store",      default=1,        type="int",    help="trace level [format: int]")
  (options, args) = parser.parse_args()

  strands        = [1, -1]
  strandToString = {1: "+", -1: "-"}
  names          = options.names.split(",")
  maxs           = {}
  nbElements     = [0 for name in names]
  lines          = [{} for i in range(len(names))]
  if options.colors == None:
    colors = [None for i in range(len(names))]
  else:
    colors = options.colors.split(",")

  parser = GffParser(options.inputFileName, options.verbosity)
  progress = Progress(parser.getNbTranscripts(), "Reading %s" % (options.inputFileName), options.verbosity)
  for transcript in parser.getIterator():
    chromosome = transcript.chromosome
    direction  = transcript.direction
    start      = transcript.start
    for i, name in enumerate(names):
      if chromosome not in lines[i]:
        lines[i][chromosome] = dict([(strand, {}) for strand in strands])
      if chromosome not in maxs:
        maxs[chromosome] = transcript.start
      else:
        maxs[chromosome] = max(maxs[chromosome], start)
      if start not in lines[i][chromosome][direction]:
        lines[i][chromosome][direction][start] = 0
      thisNbElements                          = transcript.getTagValue(name) if name in transcript.getTagNames() else 1
      lines[i][chromosome][direction][start] += thisNbElements * direction
      nbElements[i]                          += thisNbElements
    progress.inc()
  progress.done()

  if options.verbosity >= 10:
    print "Normalizing..."
  if options.normalize:
    for i, linesPerCondition in enumerate(lines):
      for linesPerChromosome in linesPerCondition.values():
        for line in linesPerChromosome.values():
          for key, value in line.iteritems():
            line[key] = value / float(nbElements[i]) * max(nbElements)
  if options.verbosity >= 10:
    print "... done."

  progress = Progress(len(maxs.keys()), "Plotting", options.verbosity)
  for chromosome in maxs:
    plot = RPlotter("%s%s.%s" % (options.outputFileName, chromosome.capitalize(), options.format), options.verbosity)
    plot.setLog(options.log)
    plot.setImageSize(20000, 5000)
    plot.setFormat(options.format)
    if maxs[chromosome] <= 1000:
      unit  = "nt."
      ratio = 1.0
    elif maxs[chromosome] <= 1000000:
      unit  = "kb"
      ratio = 1000.0
    else:
      unit  = "Mb"
      ratio = 1000000.0
    plot.setXLabel("Position on %s (in %s)" % (chromosome.replace("_", " "), unit))
    plot.setYLabel("# reads")
    plot.setLegend(True)
    for i, name in enumerate(names):
      for strand in strands:
        correctedLine = dict([(key / ratio, value) for key, value in lines[i][chromosome][strand].iteritems()])
        plot.addLine(correctedLine, "%s (%s)" % (name.replace("_", " "), strandToString[strand]), colors[i])
    plot.plot()
    progress.inc()
  progress.done()

