#! /usr/bin/env python
import random
from structure.transcriptList import *
from mySql.mySqlExonTable import *


class MySqlTranscriptTable(MySqlTable):
  """A table of transcripts in a mySQL database"""

  def __init__(self, name = None, chromosome = None, verbosity = 0):
    if chromosome == None:
      chromosome = ""
    else:
      chromosome = "_%s" % chromosome
    if name == None:
      name = "TmpTable_%d" % (random.randint(0, 100000))
    name = "%s%s" % (name, chromosome)
    self.exonsTable = MySqlExonTable(name, None, verbosity)
    super(MySqlTranscriptTable, self).__init__("%s_transcripts" % name, verbosity)


  def createTranscriptTable(self):
    self.create(Transcript.getSqlVariables(), Transcript.getSqlTypes(), Transcript.getSqlSizes())
    self.exonsTable.createExonTable()
    self.mySqlConnection.executeQuery("CREATE INDEX iBin ON %s (bin)" % self.name)
    self.mySqlConnection.executeQuery("CREATE INDEX iId ON %s (id)" % self.name)
    self.mySqlConnection.executeQuery("CREATE INDEX iStart ON %s (start)" % self.name)
    self.mySqlConnection.executeQuery("CREATE INDEX iEnd ON %s (end)" % self.name)
    self.mySqlConnection.executeQuery("CREATE INDEX iTranscriptId ON %s (transcriptId)" % self.exonsTable.name)


  def rename(self, name):
    self.exonsTable.rename(name)
    super(MySqlTranscriptTable, self).rename("%s_transcripts" % name)


  def remove(self):
    self.exonsTable.remove()
    super(MySqlTranscriptTable, self).remove()
    
    
  def clear(self):
    self.exonsTable.clear()
    super(MySqlTranscriptTable, self).clear()

    
  def copy(self, transcriptTable):
    self.remove()
    self.exonsTable.remove()
    super(MySqlTranscriptTable, self).copy(transcriptTable)
    self.exonsTable.copy(transcriptTable.exonsTable)


  def add(self, transcriptTable):
    super(MySqlTranscriptTable, self).add(transcriptTable)
    self.exonsTable.add(transcriptTable.exonsTable)


  def addTranscript(self, transcript):
    id = self.addLine(transcript.getSqlValues())
    transcript.id = id
    for exon in transcript.getExons():
      self.exonsTable.addExon(exon, id)
      
      
  def addTranscriptList(self, transcriptList):
    progress = Progress(transcriptList.getNbTranscript(), "Storing list to %s" % (self.name), self.verbosity)
    for transcript in transcriptList.getIterator():
      self.addTranscript(transcript)
      progress.inc()
    progress.done()

      
  def removeTranscript(self, transcript):
    self.removeFromId(transcript.id)
    self.exonsTable.removeFromTranscriptId(transcript.id)
      
      
  def retrieveTranscriptFromId(self, id):
    transcript = Transcript()
    transcript.setSqlValues(self.retrieveFromId(id))
    for exon in self.exonsTable.retrieveExonsFromTranscriptId(id):
      transcript.addExon(exon)
    return transcript
  
  
  def getIterator(self):
    for transcriptLine in super(MySqlTranscriptTable, self).getIterator():
      transcript = Transcript()
      transcript.setSqlValues(transcriptLine)
      for exon in self.exonsTable.retrieveExonsFromTranscriptId(transcript.id):
        transcript.addExon(exon)
      yield transcript
  
  
  def retrieveTranscriptList(self):
    transcriptList = TranscriptList()
    for transcriptLine in self.getLines():
      transcript = Transcript()
      transcript.setSqlValues(transcriptLine)
      exons = self.exonsTable.retrieveExonsFromTranscriptId(transcript.id)
      for exon in exons:
        transcript.addExon(exon)
      transcriptList.addTranscript(transcript)
    return transcriptList
      

  def setDefaultTagValue(self, name, value):
    super(MySqlTranscriptTable, self).setDefaultTagValue(Transcript.getSqlVariables().index("tags")+1, name, value)
