# Copyright INRA (Institut National de la Recherche Agronomique)
# http://www.inra.fr
# http://urgi.versailles.inra.fr
#
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software.  You can  use, 
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info". 
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability. 
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or 
# data to be ensured and,  more generally, to use and operate it in the 
# same conditions as regards security. 
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.


import unittest
import time
import os
from commons.core.sql.DbMySql import DbMySql
from commons.core.coord.Match import Match
from commons.core.sql.TableMatchAdaptator import TableMatchAdaptator


class Test_TableMatchAdaptator( unittest.TestCase ):
    
    def setUp( self ):
        self._uniqId = "%s_%s" % (time.strftime("%Y%m%d%H%M%S") , os.getpid())
        self._configFileName = "dummyConfigFile_%s" % self._uniqId
        self._iDb = DbMySql()
        self._table = "dummyMatchTable_%s" % self._uniqId
        self._tMatchA = TableMatchAdaptator( self._iDb, self._table )
        
    def tearDown( self ):
        self._uniqId = None
        self._iDb.dropTable( self._table )
        self._iDb.close()
        self._table = None
        self._tMatchA = None
        
##################################################################################
################## Tests for methods in ITableMatchAdaptator #####################
##################################################################################  
    def test_insert(self):
        match = Match()  

        tuple = ("QName1", 1, 5, 5, 0.1, 0.2, "SName1", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
       
        match.setFromTuple(tuple)
                              
        self._iDb.createTable( self._table, "match", "" )        
        self._tMatchA.insert( match, False )
        
        expTMatchTuple = (('QName1', 1L, 5L, 5L, 0.1, 0.2, 'SName1', 5L, 25L, 20L, 0.15, 1e-20, 15L, 87.2, 1L),)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._iDb.execute( sqlCmd )
        obsTmatchTuple = self._iDb.cursor.fetchall()
        
        self.assertEquals( expTMatchTuple, obsTmatchTuple )
        

    def test_insert_empty_match(self):
        match = Match()  

        tuple = ("", -1, -1, 5, 0.1, 0.2, "SName1", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
       
        match.setFromTuple(tuple)
                              
        self._iDb.createTable( self._table, "match", "" )        
        self._tMatchA.insert( match, False )
        
        expTMatchTuple = ()
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._iDb.execute( sqlCmd )
        obsTmatchTuple = self._iDb.cursor.fetchall()
        
        self.assertEquals( expTMatchTuple, obsTmatchTuple )  
               
    
    def test_insertList(self):
        match1 = Match() 
        match2 = Match()   

        tuple1 = ("QName1", 1, 5, 5, 0.1, 0.2, "SName1", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
        tuple2 = ("QName2", 2, 5, 5, 0.1, 0.2, "SName2", 6, 25, 20, 0.15, 1e-20, 15, 87.2, 2)
       
        match1.setFromTuple(tuple1)
        match2.setFromTuple(tuple2)
                              
        self._iDb.createTable( self._table, "match", "" )        
        self._tMatchA.insertList( [ match1, match2 ], False )
        
        expTMatchTuple = (('QName1', 1L, 5L, 5L, 0.1, 0.2, 'SName1', 5L, 25L, 20L, 0.15, 1e-20, 15L, 87.2, 1L),\
                          ('QName2', 2L, 5L, 5L, 0.1, 0.2, 'SName2', 6L, 25L, 20L, 0.15, 1e-20, 15L, 87.2, 2L))
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._iDb.execute( sqlCmd )
        obsTmatchTuple = self._iDb.cursor.fetchall()
        
        self.assertEquals( expTMatchTuple, obsTmatchTuple )
        
        
    def test_getMatchListFromQuery(self):
        self._iDb.createTable( self._table, "match", "" )
        tuple1 = ("QName", 1, 5, 5, 0.1, 0.2, "SName", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
        tuple2 = ("QName", 1, 6, 6, 0.2, 0.1, "SName", 6, 26, 10, 0.18, 1e-30, 18, 85.2, 2)
        tuple3 = ("QName", 1, 7, 8, 0.1, 0.2, "SName", 5, 20, 15, 0.20, 1e-25, 20, 89.0, 3)
        tuple4 = ("QName", 1, 8, 8, 0.1, 0.1, "SName", 5, 15, 10, 0.17, 1e-23, 14, 89.5, 4)
        match1 = Match()
        match1.setFromTuple( tuple1 )
        match2 = Match()
        match2.setFromTuple( tuple2 )
        match3 = Match()
        match3.setFromTuple( tuple3 )
        match4 = Match()
        match4.setFromTuple( tuple4 )
        expListMatch = [ match1, match2, match3, match4 ]
        self._tMatchA.insertList(expListMatch)
        
        obsListMatch = self._tMatchA.getMatchListFromQuery("QName")
        
        self.assertEquals(expListMatch, obsListMatch)
        
        
    def test_getMatchListFromQuery_unexisted_seq_name(self):
        self._iDb.createTable( self._table, "match", "" )
        tuple1 = ("QName", 1, 5, 5, 0.1, 0.2, "SName", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
        tuple2 = ("QName", 1, 6, 6, 0.2, 0.1, "SName", 6, 26, 10, 0.18, 1e-30, 18, 85.2, 2)
        tuple3 = ("QName", 1, 7, 8, 0.1, 0.2, "SName", 5, 20, 15, 0.20, 1e-25, 20, 89.0, 3)
        tuple4 = ("QName", 1, 8, 8, 0.1, 0.1, "SName", 5, 15, 10, 0.17, 1e-23, 14, 89.5, 4)
        match1 = Match()
        match1.setFromTuple( tuple1 )
        match2 = Match()
        match2.setFromTuple( tuple2 )
        match3 = Match()
        match3.setFromTuple( tuple3 )
        match4 = Match()
        match4.setFromTuple( tuple4 )
        lMatch = [ match1, match2, match3, match4 ]
        self._tMatchA.insertList(lMatch)
        
        expListMatch = []
        obsListMatch = self._tMatchA.getMatchListFromQuery("Dummy")
        
        self.assertEquals(expListMatch, obsListMatch)
        

    def test_getMatchListFromId(self):
        self._iDb.createTable( self._table, "match", "" )
        tuple1 = ("QName", 1, 5, 5, 0.1, 0.2, "SName", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
        tuple2 = ("QName", 1, 6, 6, 0.2, 0.1, "SName", 6, 26, 10, 0.18, 1e-30, 18, 85.2, 2)
        tuple3 = ("QName", 1, 7, 8, 0.1, 0.2, "SName", 5, 20, 15, 0.20, 1e-25, 20, 89.0, 3)
        tuple4 = ("QName", 1, 8, 8, 0.1, 0.1, "SName", 5, 15, 10, 0.17, 1e-23, 14, 89.5, 4)
        match1 = Match()
        match1.setFromTuple( tuple1 )
        match2 = Match()
        match2.setFromTuple( tuple2 )
        match3 = Match()
        match3.setFromTuple( tuple3 )
        match4 = Match()
        match4.setFromTuple( tuple4 )
        lMatch = [ match1, match2, match3, match4 ]
        expListMatch = [ match1 ]
        self._tMatchA.insertList(lMatch)
        
        obsListMatch = self._tMatchA.getMatchListFromId(1)
        
        self.assertEquals(expListMatch, obsListMatch)
        
        
    def test_getMatchListFromIdList_empty_id_list( self ):
        self._iDb.createTable( self._table, "match", "" )
        tuple1 = ("QName", 1, 5, 5, 0.1, 0.2, "SName", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
        tuple2 = ("QName", 1, 6, 6, 0.2, 0.1, "SName", 6, 26, 10, 0.18, 1e-30, 18, 85.2, 2)
        tuple3 = ("QName", 1, 7, 8, 0.1, 0.2, "SName", 5, 20, 15, 0.20, 1e-25, 20, 89.0, 3)
        tuple4 = ("QName", 1, 8, 8, 0.1, 0.1, "SName", 5, 15, 10, 0.17, 1e-23, 14, 89.5, 4)
        match1 = Match()
        match1.setFromTuple( tuple1 )
        match2 = Match()
        match2.setFromTuple( tuple2 )
        match3 = Match()
        match3.setFromTuple( tuple3 )
        match4 = Match()
        match4.setFromTuple( tuple4 )
        lMatch = [ match1, match2, match3, match4 ]
        self._tMatchA.insertList(lMatch)
        
        expList = []
        obsList = self._tMatchA.getMatchListFromIdList([])
        self.assertEquals(expList, obsList)
        
        
    def test_getMatchListFromIdList( self ):
        self._iDb.createTable( self._table, "match", "" )
        tuple1 = ("QName", 1, 5, 5, 0.1, 0.2, "SName", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
        tuple2 = ("QName", 1, 6, 6, 0.2, 0.1, "SName", 6, 26, 10, 0.18, 1e-30, 18, 85.2, 2)
        tuple3 = ("QName", 1, 7, 8, 0.1, 0.2, "SName", 5, 20, 15, 0.20, 1e-25, 20, 89.0, 3)
        tuple4 = ("QName", 1, 8, 8, 0.1, 0.1, "SName", 5, 15, 10, 0.17, 1e-23, 14, 89.5, 4)
        match1 = Match()
        match1.setFromTuple( tuple1 )
        match2 = Match()
        match2.setFromTuple( tuple2 )
        match3 = Match()
        match3.setFromTuple( tuple3 )
        match4 = Match()
        match4.setFromTuple( tuple4 )
        lMatch = [ match1, match2, match3, match4 ]
        self._tMatchA.insertList(lMatch)
        
        lObs = self._tMatchA.getMatchListFromIdList((1, 2, 3))
        
        lExp = [match1, match2, match3]
        self.assertEquals(lExp, lObs)
        
    def test_getListOfAllMatches( self ):
        self._iDb.createTable( self._table, "match", "" )
        tuple1 = ("QName", 1, 5, 5, 0.1, 0.2, "SName", 5, 25, 20, 0.15, 1e-20, 15, 87.2, 1)
        tuple2 = ("QName", 1, 6, 6, 0.2, 0.1, "SName", 6, 26, 10, 0.18, 1e-30, 18, 85.2, 2)
        tuple3 = ("QName", 1, 7, 8, 0.1, 0.2, "SName", 5, 20, 15, 0.20, 1e-25, 20, 89.0, 3)
        tuple4 = ("QName", 1, 8, 8, 0.1, 0.1, "SName", 5, 15, 10, 0.17, 1e-23, 14, 89.5, 4)
        match1 = Match()
        match1.setFromTuple( tuple1 )
        match2 = Match()
        match2.setFromTuple( tuple2 )
        match3 = Match()
        match3.setFromTuple( tuple3 )
        match4 = Match()
        match4.setFromTuple( tuple4 )
        lMatch = [ match1, match2, match3, match4 ]
        expList = [ match1, match2, match3, match4 ]
        self._tMatchA.insertList(lMatch)

        obsList = self._tMatchA.getListOfAllMatches()
        self.assertEqual( expList, obsList )
        
        
    def test_getListOfAllMatches_empty_table( self ):
        self._iDb.createTable( self._table, "match", "" )
        expList = []
        obsList = self._tMatchA.getListOfAllMatches()
        self.assertEqual( expList, obsList )
        
            
test_suite = unittest.TestSuite()
test_suite.addTest( unittest.makeSuite( Test_TableMatchAdaptator ) )
if __name__ == "__main__":
    unittest.TextTestRunner(verbosity=2).run( test_suite )
