#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#

from optparse import OptionParser
from commons.core.parsing.FastaParser import FastaParser
from SMART.Java.Python.structure.TranscriptContainer import TranscriptContainer
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc.RPlotter import RPlotter
from SMART.Java.Python.misc.Utils import *


class GetGenomeCoverage(object):

    def __init__(self, verbosity = 1):
        self.verbosity       = verbosity
        self.inputContainer  = None
        self.referenceParser = None
        self.outputFileName  = None
        self.genomeSize      = None
        self.coverage        = {}
        self.distribution    = {}


    def setInputFile(self, fileName, format):
        self.inputContainer = TranscriptContainer(fileName, format, self.verbosity)


    def setOutputFile(self, fileName):
        self.outputFileName = fileName


    def setReference(self, fileName):
        self.referenceParser = FastaParser(fileName, self.verbosity)


    def getReferenceSizes(self):
        self.genomeSize = 0
        for chromosome in self.referenceParser.getRegions():
            self.genomeSize += self.referenceParser.getSizeOfRegion(chromosome)
    

    def getCoverage(self):
        progress = Progress(self.inputContainer.getNbTranscripts(), "Reading reads", self.verbosity)
        for transcript in self.inputContainer.getIterator():
            chromosome = transcript.getChromosome()
            if chromosome not in self.coverage:
                self.coverage[chromosome] = {}
            for exon in transcript.getExons():
                for pos in range(exon.getStart(), exon.getEnd() + 1):
                    if pos not in self.coverage[chromosome]:
                        self.coverage[chromosome][pos] = 1
                    else:
                        self.coverage[chromosome][pos] += 1
            progress.inc()
        progress.done()

    
    def getDistribution(self):
        nbNucleotides = sum([len(self.coverage[chromosome].keys()) for chromosome in self.coverage])
        progress      = Progress(nbNucleotides, "Building distribution", self.verbosity)
        for chromosome in self.coverage:
            for num in self.coverage[chromosome].values():
                if num not in self.distribution:
                    self.distribution[num] = 1
                else:
                    self.distribution[num] += 1
                progress.inc()
        progress.done()
        self.distribution[0] = self.genomeSize - nbNucleotides
                

    def plotDistribution(self):
        plotter = RPlotter(self.outputFileName, self.verbosity)
        plotter.setFill(0)
        plotter.addLine(self.distribution)
        plotter.plot()
        print "min/avg/med/max reads per base: %d/%.2f/%.1f/%d" % getMinAvgMedMax(self.distribution)


    def run(self):
        self.getReferenceSizes()
        self.getCoverage()
        self.getDistribution()
        self.plotDistribution()
        

if __name__ == "__main__":
    
    # parse command line
    description = "Plot Genome Coverage v1.0.1: Get the coverage of a genome. [Category: Personal]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",     dest="inputFileName",  action="store",               type="string", help="reads file [compulsory] [format: file in transcript format given by -f]")
    parser.add_option("-f", "--format",    dest="format",         action="store",               type="string", help="format of file [compulsory] [format: transcript file format]")
    parser.add_option("-r", "--reference", dest="reference",      action="store",               type="string", help="sequences file [compulsory] [format: file in FASTA format]")
    parser.add_option("-o", "--output",    dest="outputFileName", action="store",               type="string", help="output file [compulsory] [format: output file in PNG format]")
    parser.add_option("-v", "--verbosity", dest="verbosity",      action="store", default=1,    type="int",    help="trace level [format: int]")
    (options, args) = parser.parse_args()

    getGenomeCoverage = GetGenomeCoverage(options.verbosity)
    getGenomeCoverage.setInputFile(options.inputFileName, options.format)
    getGenomeCoverage.setOutputFile(options.outputFileName)
    getGenomeCoverage.setReference(options.reference)
    getGenomeCoverage.run()
