#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""
Plot the data from the data files
"""
import os
from optparse import OptionParser
from commons.core.parsing.FastaParser import FastaParser
from commons.core.parsing.FastqParser import FastqParser
from SMART.Java.Python.misc.RPlotter import RPlotter
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc import Utils


if __name__ == "__main__":
    
    # parse command line
    description = "Get Read Distribution v1.0.1: Plot the number of identical reads and give the most represented. [Category: Visualization]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",     dest="inputFileName",  action="store",               type="string", help="input file sequence [compulsory] [format: file in sequence format given by -f]")
    parser.add_option("-f", "--format",    dest="format",         action="store",               type="string", help="format of the file [compulsory] [format: sequence file format]")
    parser.add_option("-n", "--number",    dest="number",         action="store", default=None, type="int",    help="keep the best n    [format: int]")
    parser.add_option("-p", "--percent",   dest="percent",        action="store", default=None, type="float",  help="keep the best n\% [format: float]")
    parser.add_option("-o", "--output",    dest="outputFileName", action="store",               type="string", help="output file [compulsory] [format: output files in PNG format and txt format]")
    parser.add_option("-x", "--xMax",      dest="xMax",           action="store", default=None, type="int",    help="maximum value on the x-axis to plot [format: int]")
    parser.add_option("-D", "--directory", dest="working_Dir",    action="store", default=os.getcwd(), type="string", help="the directory to store the results [format: directory]")
    parser.add_option("-v", "--verbosity", dest="verbosity",      action="store", default=1,    type="int",    help="trace level [format: int]")
    (options, args) = parser.parse_args()

    if options.working_Dir[-1] != '/':
        options.outputFileName = options.working_Dir + '/' + options.outputFileName
        
    if options.format == "fasta":
        parser = FastaParser(options.inputFileName, options.verbosity)
    elif options.format == "fastq":
        parser = FastqParser(options.inputFileName, options.verbosity)
    else:
        raise Exception("Do not understand '%s' file format." % (options.format))

    progress  = Progress(parser.getNbSequences(), "Reading %s" % (options.inputFileName), options.verbosity)
    sequences = {}
    for sequence in parser.getIterator():
        sequence = sequence.sequence
        if sequence not in sequences:
            sequences[sequence] = 1
        else:
            sequences[sequence] += 1
        progress.inc()
    progress.done()

    values = sequences.values()
    values.sort()
    if options.percent != None:
        threshold = values[int(float(options.percent) / 100 * len(values))]
    elif options.number != None:
        threshold = values[-options.number]
    else:
        threshold = 0

    # sort by value
    progress     = Progress(parser.getNbSequences(), "Sorting values", options.verbosity)
    sortedValues = dict([(value, []) for value in sequences.values()])
    for sequence, value in sequences.iteritems():
        sortedValues[value].append(sequence)
        progress.inc()
    progress.done()

    outputFileName = "%s.txt" % (options.outputFileName)
    handle         = open(outputFileName, "w")
    progress       = Progress(parser.getNbSequences(), "Writing into %s" % (outputFileName), options.verbosity)
    for value in reversed(sorted(sortedValues.keys())):
        if value >= threshold:
            for sequence in sortedValues[value]:
                handle.write("%s\t%d\n" % (sequence, value))
        progress.inc()
    progress.done()
    handle.close()

    line     = {}
    progress = Progress(len(values), "Preparing plot", options.verbosity)
    for value in values:
        if value not in line:
            line[value] = 1
        else:
            line[value] += 1
        progress.inc()
    progress.done()

    plot = RPlotter("%s.png" % (options.outputFileName), options.verbosity)
    plot.setFill(0)
    plot.setMaximumX(options.xMax)
    plot.setXLabel("# occurrences")
    plot.setYLabel("# reads")
    plot.addLine(line)
    plot.plot()

    if options.verbosity > 0:
        print "%d/%.2f/%.1f/%d occurrences" % (Utils.getMinAvgMedMax(line))


