#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""Get the size distribution of a Fasta / BED file"""

import os
from optparse import OptionParser
from commons.core.parsing.FastaParser import *
from commons.core.parsing.FastqParser import *
from SMART.Java.Python.structure.TranscriptContainer import *
from commons.core.writer.TranscriptWriter import *
from commons.core.writer.FastaWriter import *
from commons.core.writer.FastqWriter import *
from SMART.Java.Python.misc.Progress import *
from SMART.Java.Python.misc.RPlotter import *


if __name__ == "__main__":
    
    # parse command line
    description = "Restrict from Size v1.0.1: Select the elements of a list of sequences or transcripts with a given size. [Category: Data Selection]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",     dest="inputFileName",  action="store",                     type="string", help="input file [compulsory] [format: file in transcript or sequence format given by -f]")
    parser.add_option("-f", "--format",    dest="format",         action="store",                     type="string", help="format of the input [compulsory] [format: sequence or transcript file format]")
    parser.add_option("-o", "--output",    dest="outputFileName", action="store",                     type="string", help="output file [compulsory] [format: output file in transcript or sequence format given by -f]")
    parser.add_option("-m", "--minSize",   dest="minSize",        action="store",      default=None,  type="int",    help="minimum size [format: int]")
    parser.add_option("-M", "--maxSize",   dest="maxSize",        action="store",      default=None,  type="int",    help="maximum size [format: int]")
    parser.add_option("-v", "--verbosity", dest="verbosity",      action="store",      default=1,     type="int",    help="trace level [format: int]")
    parser.add_option("-l", "--log",       dest="log",            action="store_true", default=False,                help="write a log file [format: bool] [default: false]")
    (options, args) = parser.parse_args()

    if options.format == "fasta":
        parser = FastaParser(options.inputFileName, options.verbosity)
        writer = FastaWriter(options.outputFileName, options.verbosity)
    elif options.format == "fastq":
        parser = FastqParser(options.inputFileName, options.verbosity)
        writer = FastqWriter(options.outputFileName, options.verbosity)
    else:
        parser = TranscriptContainer(options.inputFileName, options.format, options.verbosity)
        writer = TranscriptWriter(options.outputFileName, options.format, options.verbosity)


    # treat items
    nbItems  = parser.getNbItems()
    progress = Progress(nbItems, "Analyzing sequences of %s" % (options.inputFileName), options.verbosity)
    nbKept   = 0
    nbRead   = 0
    nbClKept = 0
    nbClRead = 0
    for item in parser.getIterator():
        size      = item.getSize()
        nb        = 1 if options.format in ("fasta", "fastq") or "nbElements" not in item.getTagNames() else float(item.getTagValue("nbElements"))
        nbRead   += nb
        nbClRead += 1
        if (options.minSize == None or options.minSize <= size) and (options.maxSize == None or options.maxSize >= size):
            writer.addElement(item)
            nbKept   += nb
            nbClKept += 1
        progress.inc()
    progress.done()
    
    writer.write()

    print "%d items,    %d kept (%.2f%%)" % (nbRead, nbKept, 0 if nbItems == 0 else float(nbKept) / nbItems * 100)
    if nbKept != nbClKept or nbRead != nbClRead:
        print "%d clusters, %d kept (%.2f%%)" % (nbClRead, nbClKept, 0 if nbClRead == 0 else float(nbClKept) / nbClRead * 100)
