#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import re
import sys
from commons.core.parsing.MapperParser import MapperParser
from SMART.Java.Python.structure.Mapping import Mapping
from SMART.Java.Python.structure.SubMapping import SubMapping
from SMART.Java.Python.structure.Interval import Interval
from SMART.Java.Python.misc.UnlimitedProgress import UnlimitedProgress

class PslParser(MapperParser):
    """A class that parses the output of PSL format (of SSAHA and BLAT)"""

    def __init__(self, fileName, verbosity = 0):
        super(PslParser, self).__init__(fileName, verbosity)


    def __del__(self):
        super(PslParser, self).__del__()


    def getFileFormats():
        return ["psl"]
    getFileFormats = staticmethod(getFileFormats)


    def getInfos(self):
        self.chromosomes = set()
        self.nbMappings  = 0
        self.size        = 0
        self.reset()
        progress = UnlimitedProgress(100000, "Getting info on PSL file, # mappings read:", self.verbosity)
        for line in self.handle:
            progress.inc()
            line = line.strip()
            if line == "":
                continue
            parts      = line.split("\t")
            chromosome = parts[13]
            self.chromosomes.add(chromosome)
            self.nbMappings += 1
            self.size += len(parts[0])
        self.reset()
        progress.done()


    def skipFirstLines(self):
        while "------" not in self.handle.readline():
            self.currentLineNb += 1
            pass

    def _computeStarts(self,seqSize,blockSize,start,targetStrand):
        if targetStrand == "+":
            pass
        else:
            start = seqSize-blockSize-start
        return start
            


    def parseLine(self, line):
        m = re.search(r"^\s*(psl:\s+)?(\d+)\s+(\d+)\s+(\d+)\s+\d+\s+\d+\s+(\d+)\s+\d+\s+(\d+)\s+([+-]{1,2})\s+(\S+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\S+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\S+)\s+(\S+)\s+(\S+)\s*$", line)
        if m == None:
            raise Exception("\nLine %d '%s' does not have a PSL format" % (self.currentLineNb, line))

        mapping = Mapping()

        queryStrand = m.group(7)[0]

        if len(m.group(7)) == 1:
            targetStrand = "+"
        else:
            targetStrand = m.group(7)[1]


        for i in range(0, int(m.group(16))):
            size        = int(m.group(17).split(",")[i])
            queryStart  = int(m.group(18).split(",")[i])
            targetStart = int(m.group(19).split(",")[i])
            querySize   = int(m.group(9))
            targetSize  = int(m.group(13))
            
            subMapping = SubMapping()
            subMapping.setSize(size)
            subMapping.setDirection(m.group(7)[0])

            queryInterval  = Interval()
            targetInterval = Interval()

            queryInterval.setName(m.group(8))
            queryStart = self._computeStarts(querySize,size,queryStart,targetStrand)
            queryInterval.setStart(queryStart + 1)
            queryInterval.setEnd(queryStart + size)
            queryInterval.setDirection(queryStrand)

            targetInterval.setChromosome(m.group(12))
            targetStart = self._computeStarts(targetSize,size,targetStart,targetStrand)
            targetInterval.setStart(targetStart + 1)
            targetInterval.setEnd(targetStart + size)
            targetInterval.setDirection(targetStrand)

            subMapping.setQueryInterval(queryInterval)
            subMapping.setTargetInterval(targetInterval)
            mapping.addSubMapping(subMapping)

        mapping.setSize(int(m.group(2)) + int(m.group(3)) + int(m.group(4)))
        mapping.setNbMismatches(int(m.group(3)) + int(m.group(4)))
        mapping.setNbGaps(int(m.group(5)))
        mapping.setDirection(queryStrand)

        queryInterval  = Interval()
        targetInterval = Interval()

        queryInterval.setName(m.group(8))
        queryInterval.setStart(min(int(m.group(10)), int(m.group(11))))
        queryInterval.setEnd(  max(int(m.group(10)), int(m.group(11))))
        queryInterval.setDirection(queryStrand)

        targetInterval.setChromosome(m.group(12))
        targetInterval.setStart(min(int(m.group(14))+1, int(m.group(15))))
        targetInterval.setEnd(  max(int(m.group(14))+1, int(m.group(15))))
        targetInterval.setDirection(targetStrand)

        mapping.setQueryInterval(queryInterval)
        mapping.setTargetInterval(targetInterval)

        return mapping

