#! /usr/bin/env python
"""Select the transcript such that a tag value is not less than a given threshold"""

import os
from optparse import OptionParser
from structure.transcriptContainer import *
from writer.gff3Writer import *
from misc.progress import *
from misc.rPlotter import *


if __name__ == "__main__":
  
  # parse command line
  description = "Select by Tag: Keep the genomic coordinates such that a the value of a given tag is between two limits. [Category: Personnal]"

  parser = OptionParser(description = description)
  parser.add_option("-i", "--input",       dest="inputFileName",  action="store",                        type="string", help="input file [compulsory] [format: file in transcript format given by -f]")
  parser.add_option("-f", "--format",      dest="format",         action="store",                        type="string", help="format of the input [compulsory] [format: transcript file format]")
  parser.add_option("-g", "--tag",         dest="tag",            action="store",      default=None,     type="string", help="the tag [compulsory] [format: string]")   
  parser.add_option("-m", "--min",         dest="min",            action="store",      default=None,     type="float",  help="the minimum threshold [format: float]")   
  parser.add_option("-M", "--max",         dest="max",            action="store",      default=None,     type="float",  help="the maximum threshold [format: float]")   
  parser.add_option("-d", "--default",     dest="default",        action="store",      default=None,     type="float",  help="value if tag is not present [format: float]")   
  parser.add_option("-o", "--output",      dest="outputFileName", action="store",                        type="string", help="output file [format: output file in GFF3 format]")
  parser.add_option("-y", "--mysql",       dest="mysql",          action="store_true", default=False,                   help="mySQL output [format: bool] [default: false]")
  parser.add_option("-v", "--verbosity",   dest="verbosity",      action="store",      default=1,        type="int",    help="trace level [format: int]")
  parser.add_option("-l", "--log",         dest="log",            action="store_true", default=False,                   help="write a log file [format: bool] [default: false]")
  (options, args) = parser.parse_args()

  parser = TranscriptContainer(options.inputFileName, options.format, options.verbosity)

  nbWritten = 0
  writer    = Gff3Writer("%s.gff3" % (options.outputFileName), options.verbosity)
  if options.mysql:
    mysqlWriter = MySqlTranscriptWriter(options.outputFileName, options.verbosity)

  progress  = Progress(parser.getNbTranscripts(), "Writing transcripts", options.verbosity)
  for transcript in parser.getIterator():
    value = transcript.getTagValue(options.tag)
    if value == None:
      if options.default != None:
        value = options.default
      else:
        sys.exit("Error! Transcript %s no tag called '%s'" % (transcript, options.tag))
    value = float(value)
    if (options.min == None or options.min <= value) and (options.max == None or options.max >= value):
      writer.addTranscript(transcript)
      if options.mysql:
        mysqlWriter.addTranscript(transcript)
      nbWritten += 1
    progress.inc()
  progress.done()
                           
  if options.mysql:
    mysqlWriter.write()
  print "%d input" % (parser.getNbTranscripts())
  if parser.getNbTranscripts() != 0:
    print "%d output (%.2f%%)" % (nbWritten, float(nbWritten) / parser.getNbTranscripts() * 100)

