#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import re
import sys
from SMART.Java.Python.structure.Interval import Interval
from SMART.Java.Python.structure.Transcript import Transcript
from commons.core.parsing.TranscriptListParser import TranscriptListParser


class GffParser(TranscriptListParser):
	"""A class that parses a GFF file and create a transcript list"""


	def __init__(self, fileName, verbosity = 0):
		super(GffParser, self).__init__(fileName, verbosity)


	def __del__(self):
		super(GffParser, self).__del__()


	def getFileFormats():
		return ["gff", "gff2", "gff3"]
	getFileFormats = staticmethod(getFileFormats)


	def skipFirstLines(self):
		pass


	def getInfos(self):
		self.chromosomes = set()
		self.nbTranscripts = 0
		self.size = 0
		self.reset()
		if self.verbosity >= 10:
			print "Getting information on %s." % (self.fileName)
		self.reset()
		for line in self.handle:
			line = line.strip()
			if line == "" or line[0] == "#":
				continue
			parts = line.split("\t")
			if len(parts) != 9:
				raise Exception("Error! Line '%s' has %d tab-separated fields instead of 9!" % (line, len(parts)))
			self.chromosomes.add(parts[0])
			if parts[8].find("Parent") == -1:
				self.nbTranscripts += 1
			else:
				self.size += max(int(parts[3]), int(parts[4])) - min(int(parts[3]), int(parts[4])) + 1
			if self.verbosity >= 10 and self.nbTranscripts % 100000 == 0:
				sys.stdout.write("	%d transcripts read\r" % (self.nbTranscripts))
				sys.stdout.flush()
		self.reset()
		if self.verbosity >= 10:
			print "	%d transcripts read" % (self.nbTranscripts)
			print "Done."


	def parseLine(self, line):
		if not line or line[0] == "#":
			return None
		m = re.search(r"^\s*(\S+)\s+(\S+)\s+(\S+)\s+(\d+)\s+(\d+)\s+(\S+)\s+([+-.])\s+(\S+)\s+(\S.*)$", line)
		if m == None:
			raise Exception("\nLine %d '%s' does not have a GFF format\n" % (self.currentLineNb, line))
		interval = Interval()
		interval.setChromosome(m.group(1))
		interval.setName("unnamed transcript")
		interval.setStart(min(int(m.group(4)), int(m.group(5))))
		interval.setEnd(max(int(m.group(4)), int(m.group(5))))
		if m.group(7) == ".":
			interval.setDirection("+")
		else:
			interval.setDirection(m.group(7))
		interval.setTagValue("feature", m.group(3))
		if m.group(6).isdigit():
			interval.setTagValue("score", m.group(6))

		remainings = m.group(9).split(";")
		for remaining in remainings:
			remaining = remaining.strip()
			if remaining == "":
				continue
			posSpace = remaining.find(" ")
			posEqual = remaining.find("=")
			if posEqual != -1 and (posEqual < posSpace or posSpace == -1):
				parts = remaining.split("=")
			else:
				parts = remaining.split()
			field = parts[0].strip()
			value = " ".join(parts[1:]).strip(" \"")
			if field in ("Name", "name", "Sequence", "TE", "SAT"):
				interval.setName(value)
			else:
				try:
					intValue = int(value)
					interval.setTagValue(field, intValue)
				except ValueError:
					interval.setTagValue(field, value)

		self.currentTranscriptAddress = self.previousTranscriptAddress
		if "Parent" in interval.getTagNames():
			if self.currentTranscript == None:
				raise Exception("GFF file does not start with a transcript! First line is '%s'." % (line.strip()))
			if interval.getTagValue("Parent") != self.currentTranscript.getTagValue("ID"):
				raise Exception("Exon '%s' is not right after its transcript in GFF file!" % (interval))
			self.currentTranscript.addExon(interval)
			if interval.name == None:
				interval.name = self.currentTranscript.name
			return None
		
		transcript = self.currentTranscript
		self.currentTranscript = Transcript()
		self.currentTranscript.copy(interval)
		self.previousTranscriptAddress = self.currentAddress

		if transcript != None and transcript.name.startswith("unnamed"):
			if "ID" in transcript.getTagNames():
				transcript.name = transcript.getTagValue("ID")
			else:
				transcript.name = "unnamed transcript %s" % (self.currentLineNb)
		return transcript
