#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""Trim the sequences from a 5' adaptor"""

import sys
from optparse import OptionParser
from commons.core.parsing.FastaParser import FastaParser
from commons.core.parsing.FastqParser import FastqParser
from commons.core.writer.FastaWriter import FastaWriter
from commons.core.writer.FastqWriter import FastqWriter
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc import Utils


if __name__ == "__main__":
    
    # parse command line
    description = "Trim Adaptor v1.0.1: Remove the 3' adaptor of a list of reads. [Category: Data Modification]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",     dest="inputFileName",  action="store",               type="string", help="input file [compulsory] [format: file in sequence format given by -f]")
    parser.add_option("-f", "--format",    dest="format",         action="store",               type="string", help="format of file [compulsory] [format: sequence file format]")
    parser.add_option("-o", "--output",    dest="outputFileName", action="store",               type="string", help="output file [compulsory] [format: output file in sequence format given by -f]")
    parser.add_option("-a", "--adaptor",   dest="adaptor",        action="store",               type="string", help="adaptor [compulsory] [format: string]")
    parser.add_option("-e", "--errors",    dest="errors",         action="store", default=0,    type="int" ,   help="number of errors in percent [format: int] [default: 0]")
    parser.add_option("-n", "--noAdaptor", dest="noAdaptor",      action="store", default=None, type="string", help="file name where to print sequences with no adaptor [format: output file in sequence format given by -f]")
    parser.add_option("-v", "--verbosity", dest="verbosity",      action="store", default=1,    type="int",    help="trace level [format: int]")
    (options, args) = parser.parse_args()

    minSize = 2

    if options.format == "fasta":
        parser = FastaParser(options.inputFileName, options.verbosity)
    elif options.format == "fastq":
        parser = FastqParser(options.inputFileName, options.verbosity)
    else:
        sys.exit("Cannot handle files with '%s' format." % (options.format))

    if options.format == "fasta":
        writer = FastaWriter(options.outputFileName, options.verbosity)
    elif options.format == "fastq":
        writer = FastqWriter(options.outputFileName, options.verbosity)
    else:
        sys.exit("Cannot handle files with '%s' format." % (options.format))

    writerNoAdaptor = None
    if options.noAdaptor != None:
        if options.format == "fasta":
            writerNoAdaptor = FastaWriter(options.noAdaptor, options.verbosity)
        elif options.format == "fastq":
            writerNoAdaptor = FastqWriter(options.noAdaptor, options.verbosity)
        else:
            sys.exit("Cannot handle files with '%s' format." % (options.format))

    nbFound = 0
        
    progress = Progress(parser.getNbSequences(), "Reading %s" % (options.inputFileName), options.verbosity)
    for sequence in parser.getIterator():
        progress.inc()
        nucleotides = sequence.getSequence()
        found       = False
        for i in range(len(nucleotides) - minSize):
            nucleotidesPart = nucleotides[i:]
            adaptorPart     = options.adaptor if len(nucleotidesPart) >= len(options.adaptor) else options.adaptor[:len(nucleotidesPart)]
            nucleotidesPart = nucleotidesPart if len(adaptorPart) == len(nucleotidesPart) else nucleotidesPart[:len(adaptorPart)]
            if Utils.getHammingDistance(adaptorPart, nucleotidesPart) <= int(options.errors / 100.0 * len(adaptorPart)):
                nbFound += 1
                sequence.shrinkToFirstNucleotides(i)
                writer.addSequence(sequence)
                found = True
                break
        if not found:
            writer.addSequence(sequence)
            if writerNoAdaptor != None:
                writerNoAdaptor.addSequence(sequence)
    progress.done()

    print "%d sequences with adaptors on %d (%.2f%%)" % (nbFound, parser.getNbSequences(), float(nbFound) / parser.getNbSequences() * 100)

