#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
from commons.core.writer.TranscriptListWriter import TranscriptListWriter


class EmblWriter(TranscriptListWriter):
    """
    A class that writes a transcript list into several files with EMBL format
    @ivar fileName: name of the file 
    @type fileName: string
    @ivar handle: handle to the file
    @type handle: file handle
    """


    def __init__(self, fileName, verbosity = 0):
        """
        Constructor
        @param fileName: name of the file 
        @type fileName: string
        @param verbosity: verbosity
        @type verbosity: int
        """
        self.fileName = fileName
        self.verbosity = verbosity
        self.handles = {}
        self.handle = None


    def __del__(self):
        """
        Destructor
        Trick to append the sequences at the end of the EMBL files
        """
        handle                = open(self.sequenceFileName)
        currentHandle = None
        for line in handle:
            if line[0] == ">":
                chromosome = line[1:].strip()
                if chromosome in self.handles:
                    currentHandle = self.handles[chromosome]
                else:
                    currentHandle = None
            else:
                if currentHandle != None:
                    currentHandle.write(line)
        handle.close()
        for handle in self.handles.values():
            handle.close()
            

    @staticmethod
    def getFileFormats():
        """
        Get the format of the file
        """
        return ["embl"]
        
        
    @staticmethod
    def getExtension():
        """
        Get the usual extension for the file
        """
        return "embl"


    def addTranscript(self, transcript):
        """
        Add a transcript to the list of transcripts to be written
        @param transcript: transcript to be written
        @type    transcript: class L{Transcript<Transcript>}
        """
        chromosome = transcript.getChromosome()
        if chromosome not in self.handles:
            self.handles[chromosome] = open("%s%s.embl" % (self.fileName[:-len(".embl")], chromosome.title()), "w")
        self.handles[chromosome].write(self.printTranscript(transcript))


    def printTranscript(self, transcript):
        """
        Export the given transcript with GFF2 format
        @param transcript: transcript to be printed
        @type transcript: class L{Transcript<Transcript>}
        @return: a string
        """
        return transcript.printEmbl()


